/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.JCRProjectMember;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Notifier to send mail to a newly added member of a workspace.
 */
public class WelcomeMemberMailNotifierObserver extends AbstractMemberMailNotifierObserver
{
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_MEMBER_ADDED)
                && Config.getInstance().getValue("workspaces.member.added.send.notification", true, false); // Check feature enabled
    }

    @Override
    protected Map<String, List<String>> getUserToNotifyByLanguage(Event event, Project project)
    {
        // async observer, we need to resolve the member
        String memberId = (String) event.getArguments().get(ObservationConstants.ARGS_MEMBER_ID);
        JCRProjectMember newMember = _resolver.resolveById(memberId);
        
        String defaultLanguage = _userLanguagesManager.getDefaultLanguage();
        
        // get user to notify
        if (MemberType.USER == newMember.getType())
        {
            User user = _userManager.getUser(newMember.getUser());
            String email = user.getEmail();
            if (StringUtils.isNotEmpty(email) && _isUserMemberNewToProject(user.getIdentity(), project))
            {
                return Collections.singletonMap(StringUtils.defaultIfBlank(user.getLanguage(), defaultLanguage), List.of(email));
            }
        }
        else if (MemberType.GROUP == newMember.getType())
        {
            GroupIdentity groupIdentity = newMember.getGroup();
            Group group = _groupManager.getGroup(groupIdentity);
            if (group != null && project != null)
            {
                List<User> newUsersInProject = _projectMemberManager.getGroupUsersFromProject(group, project, (currentProject, identity) -> _isGroupMemberNewToTheProject(currentProject, identity, groupIdentity));
                
                return newUsersInProject.stream()
                        .map(user -> Pair.of(user, user.getEmail()))
                        .filter(p -> StringUtils.isNotEmpty(p.getRight()))
                        .collect(Collectors.groupingBy(
                                p -> {
                                    return StringUtils.defaultIfBlank(p.getLeft().getLanguage(), defaultLanguage);
                                },
                                Collectors.mapping(
                                        Pair::getRight,
                                        Collectors.toList()
                                )
                            )
                        );
            }
        }
        
        return Map.of();
    }
    
    @Override
    protected I18nizableText getI18nSubject(Event event, Project project)
    {
        return new I18nizableText("plugin." + _pluginName, "PROJECT_MAIL_NOTIFICATION_EVENT_SUBJECT_MEMBER_ADDED", List.of(project.getTitle()));
    }
    
    @Override
    protected String getMailBodyURI(Event event, Project project)
    {
        return "cocoon://_plugins/workspaces/notification-mail-member-event";
    }

    private boolean _isGroupMemberNewToTheProject(Project project, UserIdentity userIdentity, GroupIdentity newGroupToIgnore)
    {
        Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, false);
        boolean isUserOfProject = projectMembers.stream()
                .filter(member -> MemberType.USER == member.getType())
                .map(ProjectMember::getUser)
                .map(User::getIdentity)
                .filter(userIdentity::equals)
                .findAny()
                .isPresent();
        
        if (isUserOfProject)
        {
            // User from group was already part of the project, as a member of type "user"
            return false;
        }
        
        return _isUserMemberInAGroupMember(userIdentity, newGroupToIgnore, projectMembers);
    }
    
    private boolean _isUserMemberNewToProject(UserIdentity userIdentity, Project project)
    {
        Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, false);
        return _isUserMemberInAGroupMember(userIdentity, null, projectMembers);
    }

    private boolean _isUserMemberInAGroupMember(UserIdentity userIdentity, GroupIdentity newGroupToIgnore, Set<ProjectMember> projectMembers)
    {
        return projectMembers.stream()
                .filter(member -> MemberType.GROUP == member.getType())
                .map(ProjectMember::getGroup)
                .filter(group -> newGroupToIgnore == null || !newGroupToIgnore.equals(group.getIdentity()))
                .filter(Objects::nonNull)
                .map(Group::getUsers)
                .flatMap(Set::stream)
                .filter(userIdentity::equals)
                .findAny()
                .isEmpty(); // User was not found in any group of the project, apart from the ignored group
    }
}
