/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tags;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.AbstractTagProvider;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.DefaultTag;
import org.ametys.cms.tag.TagTargetType;
import org.ametys.cms.tag.TagTargetTypeExtensionPoint;

/**
 * Class representing projects' tags as CMS tag provider.
 * This directly use {@link ProjectJCRTagProvider} and transform the projects' tags in {@link CMSTag}
 */
public class ProjectCMSTagProvider extends AbstractTagProvider<CMSTag> implements Serviceable
{
    /** Target types */
    protected TagTargetTypeExtensionPoint _targetTypeEP;
    
    /** Category tag provider */
    protected ProjectTagProviderExtensionPoint _projectTagProviderEP;
    
    /** The JCR category provider */
    protected ProjectJCRTagProvider _jcrProjectTagProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _targetTypeEP = (TagTargetTypeExtensionPoint) smanager.lookup(TagTargetTypeExtensionPoint.ROLE);
        _projectTagProviderEP = (ProjectTagProviderExtensionPoint) smanager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _jcrProjectTagProvider = (ProjectJCRTagProvider) _projectTagProviderEP.getExtension(ProjectJCRTagProvider.class.getName());
    }
    
    public Map<String, CMSTag> getTags(Map<String, Object> contextualParameters)
    {
        Map<String, CMSTag> cmsTags = new HashMap<>();
        
        Map<String, DefaultTag> projectTags = _jcrProjectTagProvider.getTags(contextualParameters);
        
        for (Entry<String, DefaultTag> projectTag : projectTags.entrySet())
        {
            ProjectCMSTag cmsTag = _transformToCMSTag(projectTag.getValue(), contextualParameters);
            cmsTags.put(cmsTag.getId(), _transformToCMSTag(projectTag.getValue(), contextualParameters));
        }
        
        return cmsTags;
    }
    
    public CMSTag getTag(String tagName, Map<String, Object> contextualParameters)
    {
        String projectTagName = tagName;
        if (projectTagName.startsWith(ProjectCMSTag.TAG_PREFIX))
        {
            projectTagName = StringUtils.substringAfter(tagName, ProjectCMSTag.TAG_PREFIX);
            return _transformToCMSTag(_jcrProjectTagProvider.getTag(projectTagName, contextualParameters), contextualParameters);
        }
        return null;
    }
    
    public Collection<CMSTag> getTags(String tagName, Map<String, Object> contextualParameters)
    {
        String projectTagName = tagName;
        if (projectTagName.startsWith(ProjectCMSTag.TAG_PREFIX))
        {
            projectTagName = StringUtils.substringAfter(tagName, ProjectCMSTag.TAG_PREFIX);
            Collection<DefaultTag> projectTags = _jcrProjectTagProvider.getTags(projectTagName, contextualParameters);
            
            return projectTags.stream()
                    .map(tag -> _transformToCMSTag(tag, contextualParameters))
                    .collect(Collectors.toList());
        }
        return List.of();
    }
    
    public boolean hasTag(String tagName, Map<String, Object> contextualParameters)
    {
        String projectTagName = tagName;
        if (projectTagName.startsWith(ProjectCMSTag.TAG_PREFIX))
        {
            projectTagName = StringUtils.substringAfter(tagName, ProjectCMSTag.TAG_PREFIX);
            return _jcrProjectTagProvider.hasTag(projectTagName, contextualParameters);
        }
        return false;
    }
    
    /**
     * Transform a project's tag in a {@link CMSTag}, forcing visibility to public and target to CONTENT
     * @param projectTag the project tag to transform
     * @param contextualParameters the contextual parameters
     * @return a {@link CMSTag} with the same values
     */
    protected ProjectCMSTag _transformToCMSTag(DefaultTag projectTag, Map<String, Object> contextualParameters)
    {
        if (projectTag != null)
        {
            CMSTag parentTag = null;
            DefaultTag parentProjectTag = projectTag.getParent();
            if (parentProjectTag != null)
            {
                parentTag = this.getTag(parentProjectTag.getName(), contextualParameters);
            }
            
            TagTargetType targetType = _targetTypeEP.getTagTargetType("CONTENT");
            
            return new ProjectCMSTag(projectTag, parentTag, targetType);
        }
        else
        {
            return null;
        }
    }
}
