/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tags;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.tag.AbstractTagsDAO;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.cms.tag.jcr.AbstractJCRTagsDAO;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * DAO for manipulating project's tags
 */
public class ProjectTagsDAO extends AbstractTagsDAO
{
    /** The avalon role */
    public static final String ROLE = ProjectTagsDAO.class.getName();
    
    /** The right manager */
    protected RightManager _rightManager;

    /** The project JCR tags DAO */
    protected ProjectJCRTagsDAO _projectJCRDAO;
    
    /** The workspace module extension point */
    protected WorkspaceModuleExtensionPoint _workspaceModuleEP;
    
    /** The project manager */
    protected ProjectManager _projectManager;

    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The project right helper */
    protected ProjectRightHelper _projectRightHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _projectJCRDAO = (ProjectJCRTagsDAO) manager.lookup(ProjectJCRTagsDAO.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _projectRightHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        
    }
    
    @Override
    public String getTagProviderEPRole()
    {
        return ProjectTagProviderExtensionPoint.ROLE;
    }

    @Override
    protected List<TagProvider< ? extends Tag>> getCustomTagProvider()
    {
        List<TagProvider<? extends Tag>> providers = new ArrayList<>();
        providers.add(_tagProviderExtPt.getExtension(ProjectJCRTagProvider.class.getName()));
        
        return providers;
    }
    
    /**
     * Get existing project tags
     * @param moduleId the module id
     * @param projectName the project name
     * @return the tags (id and label)
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getProjectTags(String moduleId, String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        if (!_projectRightHelper.hasReadAccessOnModule(project, moduleId))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }
        
        List<Map<String, Object>> tags = new ArrayList<>();
        
        for (String id : _tagProviderExtPt.getExtensionsIds())
        {
            TagProvider<? extends Tag> tagProvider = _tagProviderExtPt.getExtension(id);
            Map<String, ? extends Tag> someTags = tagProvider.getTags(Collections.emptyMap());

            tags.addAll(_tagsToJSON(someTags.values()));
        }

        Map<String, Object> result = new HashMap<>();
        result.put("tags", tags);
        
        WorkspaceModule module = _workspaceModuleEP.getModule(moduleId);
        result.put("canCreate", _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_ADD_TAG, module.getModuleRoot(project, false)) == RightResult.RIGHT_ALLOW);
        
        return result;
    }
    
    @Override
    protected AbstractJCRTagsDAO _getTagJCRDAO()
    {
        return _projectJCRDAO;
    }
}
