/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.wall;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.excalibur.xml.sax.SAXParser;
import org.apache.solr.client.solrj.SolrServerException;
import org.xml.sax.InputSource;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.content.RichTextHandler;
import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.Binary;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ResourceElementTypeHelper;
import org.ametys.cms.indexing.IndexingObserver;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentDAO.TagMode;
import org.ametys.cms.repository.comment.ui.CommentsAndReportsTreeComponent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.observation.ObservationManager.ObserverFuture;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.upload.Upload;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.content.FOContentCreationHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * Helper for wall contents
 *
 */
public class WallContentManager extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = WallContentManager.class.getName();
    /** The tag for pin */
    public static final String WALL_CONTENT_PIN_TAG = "WORKSPACES_CONTENT_PINNED";
    
    private static final int __INITIAL_WORKFLOW_ACTION_ID = 1111;
    private static final String __WORKFLOW_NAME = "wall-content";
    
    private FOContentCreationHelper _foContentHelper;
    private ContentTypeExtensionPoint _cTypeEP;
    private I18nUtils _i18nUtils;
    private ContentDAO _contentDAO;
    private ObservationManager _observationManager;
    private RightManager _rightManager;
    private AmetysObjectResolver _resolver;
    private UserManager _userManager;
    private CurrentUserProvider _currentUserProvider;
    private ProjectManager _projectManager;
    private SiteManager _siteManager;
    private ServiceManager _smanager;
    private UploadManager _uploadManager;
    private SolrIndexer _solrIndexer;
    private WorkspaceModuleExtensionPoint _moduleEP;
    private CommentsAndReportsTreeComponent _commentAndReportCmp;
    private ProjectRightHelper _projectRightHelper;
    private UserLanguagesManager _userLanguagesManager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _smanager = manager;
        _foContentHelper = (FOContentCreationHelper) manager.lookup(FOContentCreationHelper.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _uploadManager = (UploadManager) manager.lookup(UploadManager.ROLE);
        _solrIndexer = (SolrIndexer) manager.lookup(SolrIndexer.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _commentAndReportCmp = (CommentsAndReportsTreeComponent) manager.lookup(CommentsAndReportsTreeComponent.ROLE);
        _projectRightHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _userLanguagesManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
    }
    
    /**
     * Create and publish a new wall content
     * @param text the text
     * @param part the file part for illustration. Can be null.
     * @param siteName the site name
     * @param lang the language
     * @return the results
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> publishContent(String text, Part part, String siteName, String lang)
    {
        Map<String, Object> results = new HashMap<>();

        try
        {
            if (!_projectRightHelper.hasReadAccess())
            {
                throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to publish wall content on non authorized project");
            }
            
            ContentType cType = _cTypeEP.getExtension(WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID);
            
            String contentTitle = _i18nUtils.translate(cType.getDefaultTitle(), lang);
            
            Map<String, Object> userValues = new HashMap<>();
            userValues.put(Content.ATTRIBUTE_TITLE, contentTitle);
            userValues.put("content", text);
            userValues.put("comment", true); // active comments
            
            if (part != null)
            {
                try (InputStream is = part.getInputStream())
                {
                    Upload upload = _uploadManager.storeUpload(_currentUserProvider.getUser(), part.getFileName(), is);
                    Binary fileValue = ResourceElementTypeHelper.binaryFromUpload(upload);
                    
                    userValues.put("illustration", Map.of("image", fileValue));
                }
                catch (IOException e)
                {
                    getLogger().error("Failed to store uploaded wall content illustration", e);
                }
            }
            
            try
            {
                results = _foContentHelper.createAndEditContent(__INITIAL_WORKFLOW_ACTION_ID, WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID, siteName, contentTitle, contentTitle, lang, userValues, __WORKFLOW_NAME, null);
                
                Content content = (Content) results.get(Content.class.getName());
                _notifyContentCreation(content);
                
                // remove Content from result
                results.remove(Content.class.getName());
                
                results.put("success", true);
            }
            finally
            {
                _commitAllChanges();
            }
        }
        catch (AmetysRepositoryException | WorkflowException e)
        {
            results.put("success", false);
            getLogger().error("Failed to create wall content for site {} and language {}", siteName, lang, e);
        }
        return results;
    }
    
    /**
     * Commit all changes in solr
     */
    protected void _commitAllChanges()
    {
        // Before trying to commit, be sure all the async observers of the current request are finished
        List<ObserverFuture> futuresForRequest = _observationManager.getFuturesForRequest();
        for (ObserverFuture observerFuture : futuresForRequest)
        {
            if (observerFuture.traits().contains(IndexingObserver.INDEXING_OBSERVER))
            {
                try
                {
                    observerFuture.future().get();
                }
                catch (ExecutionException | InterruptedException e)
                {
                    getLogger().info("An exception occured when calling #get() on Future result of an observer." , e);
                }
            }
        }
        
        // Force commit all uncommited changes
        try
        {
            _solrIndexer.commit();
        }
        catch (IOException | SolrServerException e)
        {
            getLogger().error("Impossible to commit changes", e);
        }
    }
    
    private void _notifyContentCreation(Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.ObservationConstants.EVENT_WALLCONTENT_ADDED, content.getCreator(), eventParams));
    }
    
    /**
     * Pin a wall content
     * @param contentId the content id
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> pinContent(String contentId, Map<String, Object> contextualParameters)
    {
        if (!_projectRightHelper.hasRightOnModule("Plugins_Workspaces_Right_Pin_WallContent", WallContentModule.WALLCONTENT_MODULE_ID))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to pin content '" + contentId + "' without convenient right.");
        }
        
        return _pinOrUnpinContent(contentId, contextualParameters, TagMode.INSERT);
    }
    
    /**
     * Unpin a wall content
     * @param contentId the content id
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> unpinContent(String contentId, Map<String, Object> contextualParameters)
    {
        if (!_projectRightHelper.hasRightOnModule("Plugins_Workspaces_Right_Pin_WallContent", WallContentModule.WALLCONTENT_MODULE_ID))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to unpin content '" + contentId + "' without convenient right.");
        }
        
        return _pinOrUnpinContent(contentId, contextualParameters, TagMode.REMOVE);
    }
    
    private Map<String, Object> _pinOrUnpinContent(String contentId,  Map<String, Object> contextualParameters, TagMode mode)
    {
        try
        {
            Map<String, Object> result = _contentDAO.tag(Collections.singletonList(contentId), Collections.singletonList(WALL_CONTENT_PIN_TAG), mode, contextualParameters, true);
            return result;
        }
        finally
        {
            _commitAllChanges();
        }
    }
    
    /**
     * Report content to webmasters (user with report notification right on wall contents)
     * @param siteName the current site name
     * @param contentId the id of content to report
     * @return true if the content was successfully reported
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 1, rightContext = ContentRightAssignmentContext.ID)
    public boolean reportContent(String siteName, String contentId)
    {
        Content content = _resolver.resolveById(contentId);
        User reporter = _userManager.getUser(_currentUserProvider.getUser());
        Site site = _siteManager.getSite(siteName);
        
        // Add the report to the content
        _contentDAO.report(content);
        
        // Send a mail to the allowed users
        List<Project> projects = _projectManager.getProjectsForSite(site);
        if (!projects.isEmpty())
        {
            Project project = projects.get(0);
            
            Map<String, List<String>> recipientsByLanguage = _getReportsRecipientsByLanguage(project);
            if (!recipientsByLanguage.isEmpty())
            {
                Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                i18nParams.put("projectTitle", new I18nizableText(project.getTitle()));
                i18nParams.put("siteTitle", new I18nizableText(site.getTitle()));
                
                I18nizableText i18nSubject = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_WALL_CONTENT_REPORTED_SUBJECT", i18nParams);
                
                i18nParams.put("projectUrl", new I18nizableText(site.getUrl()));
                i18nParams.put("reporter", new I18nizableText(reporter.getFullName()));
                i18nParams.put("content", new I18nizableText(getExcerpt(content, 200)));
                
                String from = site.getValue("site-mail-from");
                
                MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                        .withTitle(i18nSubject)
                        .withMessage(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_WALL_CONTENT_REPORTED_BODY", i18nParams))
                        .withDetails(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_WALL_CONTENT_REPORTED_BODY_EXTRACT"), getExcerpt(content, 200), false)
                        .withLink(site.getUrl(), new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_DEFAULT_BUTTON_TEXT"));
                
                for (String userLanguage : recipientsByLanguage.keySet())
                {
                    try
                    {
                        List<String> emails = recipientsByLanguage.get(userLanguage);
                        String subject = _i18nUtils.translate(i18nSubject, userLanguage);
                        
                        String htmlBody = bodyBuilder.withLanguage(userLanguage).build();
                        
                        SendMailHelper.newMail()
                            .withSubject(subject)
                            .withHTMLBody(htmlBody)
                            .withSender(from)
                            .withRecipients(emails)
                            .sendMail();
                    }
                    catch (MessagingException | IOException e)
                    {
                        getLogger().warn("Could not send a notification mail to {}", recipientsByLanguage.get(userLanguage), e);
                    }
                }

                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get the excerpt of content
     * @param content the content
     * @param maxLength the max length for content excerpt
     * @return the excerpt
     */
    public String getExcerpt(Content content, int maxLength)
    {
        if (content.hasValue("content"))
        {
            RichText richText = content.getValue("content");
            SAXParser saxParser = null;
            try (InputStream is = richText.getInputStream())
            {
                RichTextHandler txtHandler = new RichTextHandler(maxLength);
                saxParser = (SAXParser) _smanager.lookup(SAXParser.ROLE);
                saxParser.parse(new InputSource(is), txtHandler);
                
                return txtHandler.getValue();
            }
            catch (Exception e)
            {
                getLogger().error("Cannot extract excerpt from content {}", content.getId(), e);
            }
            finally
            {
                _smanager.release(saxParser);
            }
        }
     
        return "";
    }
    
    /**
     * Retrieves the list of recipients for reports notification sending
     * @param project the current project
     * @return the list of recipients for reports notification sending
     */
    protected Map<String, List<String>> _getReportsRecipientsByLanguage(Project project)
    {
        WorkspaceModule module = _moduleEP.getModule(WallContentModule.WALLCONTENT_MODULE_ID);
        ResourceCollection moduleRoot = module.getModuleRoot(project, false);
        
        Set<UserIdentity> users = _rightManager.getAllowedUsers("Plugins_Workspaces_Right_ReportNotification_WallContent", moduleRoot).resolveAllowedUsers(Config.getInstance().getValue("runtime.mail.massive.sending"));
        
        String defaultLanguage = _userLanguagesManager.getDefaultLanguage();
        
        return users.stream()
                .map(_userManager::getUser)
                .filter(user -> user != null)
                .map(user -> Pair.of(user, user.getEmail()))
                .filter(p -> StringUtils.isNotBlank(p.getRight()))
                .collect(Collectors.groupingBy(
                        p -> {
                            return StringUtils.defaultIfBlank(p.getLeft().getLanguage(), defaultLanguage);
                        },
                        Collectors.mapping(
                                Pair::getRight,
                                Collectors.toList()
                        )
                    )
                );
    }
    
    /**
     * Add or remove a reaction on a content
     * @param contentId The content id
     * @param reactionName the reaction name (ex: LIKE)
     * @param remove true to remove the reaction, false to add reaction
     * @return the result with the current actors of this reaction
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> react(String contentId, String reactionName, boolean remove)
    {
        return _contentDAO.react(contentId, reactionName, remove);
    }
    
    /**
     * Retrieves the comments and reports of the contents of the given type
     * Manages only contents that have at least one report (on itself or on a comment)
     * @param contentTypeId the content type identifier
     * @return the comments and reports of the contents
     * @throws RepositoryException if an error occurs while retrieving contents from the repository
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map getCommentsAndReportsFromContentTypeId(String contentTypeId) throws RepositoryException
    {
        return _commentAndReportCmp.getCommentsAndReportsFromContentTypeId(contentTypeId, "Plugins_Workspaces_Right_See_Reports_WallContent");
    }
    
}
