/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

// Register old nodetypes, as they need to be registered to be removed
Repository.registerNodeType("post");
Repository.registerNodeType("thread");

const unversionedDataTypeExtensionPoint = Ametys.serviceManager.lookup('org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint.Unversioned');
const richTextTransformer = Ametys.serviceManager.lookup('org.ametys.cms.transformation.docbook.DocbookTransformer');
const workspaceExplorerResourceCommentsDAO = Ametys.serviceManager.lookup('org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceCommentsDAO');
const userManager = Ametys.serviceManager.lookup(org.ametys.core.user.UserManager.ROLE);

logger.info("Start content types migration for default session");
convertRootFolder(Repository.session);

logger.info("Start content types migration for live session");
convertRootFolder(Repository.liveSession);

logger.info("Start comment migration for default session");
convertComments(Repository.session);

logger.info("Start comment migration for live session");
convertComments(Repository.liveSession);

function convertRootFolder(session) 
{
    let count = 0;
    const xpath = "//element(*, ametys:project)/ametys-internal:resources/documents";
    const activityNodes = Repository.query(xpath, false, session);
            
    while (activityNodes.hasNext())
    {
        let node = activityNodes.next();
        convertNode(node)
        count++;
    }
    logger.info(`${count} project with reporitory resources have been migrated in ${session.getWorkspace().getName()} workspace`);
    session.save();
}


function convertNode(node) 
{
    let primaryType = node.getPrimaryNodeType().getName();
    if (primaryType == "ametys:resources-collection")
    {
        node.setPrimaryType("ametys:folder");
        node.save();
        let children = node.getNodes();
        while (children.hasNext())
        {
            let child = children.next();
            convertNode(child);
        }
    }
    else if (primaryType == "ametys:resource")
    {
        node.setPrimaryType("ametys:file");
        node.save();
    }
}

function convertComments(session) 
{
    
    const xpath = "//element(*, ametys:file)";
    Repository.query(xpath, true, session).forEach(function(resource) 
    {
        let resourceNode = resource.getNode();
    
        if (resourceNode.hasNode("ametys:comments")) 
        {
            let commentsNode = resourceNode.getNode("ametys:comments");
    
            let children = commentsNode.getNodes();
            let commentsToMigrate = []
            while (children.hasNext()) 
            {
                let child = children.next();
                let childType = child.getPrimaryNodeType().getName();
                if (childType == "ametys:post")
                {
                    let contentNode = child.getNode("ametys:content");
    
                    let repositoryData = new org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData(child);
                    let dataholder = new org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder(unversionedDataTypeExtensionPoint, repositoryData);
    
                    let richText = dataholder.getValueOfType("content", org.ametys.cms.data.type.ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID);
                    let result = new java.lang.StringBuilder(2048);
                    richTextTransformer.transformForEditing(richText, org.ametys.runtime.model.type.DataContext.newInstance(), result);
    
                    let creationDate = child.getProperty("ametys:creationDate").getValue();
                    let creationDateAsZDT = org.ametys.core.util.DateUtils.asZonedDateTime(creationDate.getDate())
                  
                    let authorNode = child.getNode("ametys:author");
                    let login = authorNode.getProperty("ametys:login").getString();
                    let population = authorNode.getProperty("ametys:population").getString();
                    let commentToMigrate = 
                    {
                        userIdentity: new org.ametys.core.user.UserIdentity(login, population),
                        text: result.toString(),
                        creationDate: creationDateAsZDT
                    }
                    commentsToMigrate.push(commentToMigrate);
                }
            }
            
            // We need to remove the node to recreate it with a good type
            commentsNode.remove() 
            
            for (commentToMigrate of commentsToMigrate)
            {          
                let comment = resource.createComment(null, commentToMigrate.creationDate);
                let userIdentity = commentToMigrate.userIdentity;
                let user = userManager.getUser(userIdentity);
              
                if (user != null)
                {
                    comment.setAuthorName(user.getFullName());
                    comment.setAuthorEmail(user.getEmail());
                }
                comment.setAuthor(userIdentity);
                comment.setEmailHiddenStatus(true);
                comment.setContent(commentToMigrate.text);
            }
            resource.saveChanges();
            
        }
    
    });
    
}
