/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Dialog box used to select one or more tags
 * @private
 */
Ext.define('Ametys.plugins.workspaces.project.helper.ChooseTag', {
    singleton: true,

    /**
     * @property {Object} initialConfig The initial configuration object passed to the #open method:
     * @property {Boolean} [initialConfig.multiple=false] True to allow multiple selections
     * @property {Boolean} [initialConfig.onlyCustomTags=false] If true, only custom tags (JCR) are displayed
     * @property {Boolean} [initialConfig.onlyTagsWithChildren=false] If 'true', only tags with children can be selected.
     * @property {Boolean} [initialConfig.allowProviders=false] If 'true', tag providers are also be selected.
     * @property {Boolean} [initialConfig.allowCreation=false] If 'true', a link at the dialog bottom proposes to add a new tag (defaults to false).
     * @property {String} [initialConfig.targetType] The type of tags to display. If set to a value, display the tags of this target type. If set to null, display all tags.
     * @property {String} [initialConfig.filterTarget] Filter on target type. If set to a value, disable the tags of this target type. If set to null, enable all tags.
     * @property {String} [initialConfig.url="tags.json"] The url to get the tags informations
     * @property {String} [initialConfig.icon] The full path to icon (16x16 pixels) for the dialog box.
     * @property {String} [initialConfig.iconCls="ametysicon-label49"] The separated CSS class to apply for dialog icon. Use only if the 'icon' configuration is undefined.
     * @property {String} initialConfig.title The title of dialog box
     * @property {String} [initialConfig.helpMessage] The help message to display on top of dialog box.
     * @property {String[]} [initialConfig.values] the selected tags as an array of id.
     * @property {String[]} initialConfig.taggableObjects the ids of current objects to be tagged.
     * @property {Object} initialConfig.contextParameters the context parameters to add to callback parameters
     * @property {String[]} initialConfig.callback The callback function invoked when tags are selected. The callback function will received the following parameters:
     * @property {Function} initialConfig.callback.tags The names of selected tags
     * @property {Object} initialConfig.callback.params The callback additional parameters
     */

    /**
     * @private
     * @property {Ametys.plugins.workspaces.project.helper.TagsTreePanel} _tree The tags tree panel
     */

    /**
     * Configure and open the dialog box
     * @param {Object} config The configuration options :
     * @param {String} [config.url=tags.json] The url to get the tags informations.
     * @param {String} [config.iconCls] The CSS class for the icon of the dialog box.
     * @param {String} [config.title] The title of the dialog box.
     * @param {String} [config.helpMessage] The help message to display on top of dialog box.
     * @param {String[]} [config.values] the selected tags as an array of id.
     * @param {String[]} [config.taggableObjects] the ids of current objects to be tagged.
     * @param {Boolean} [config.multiple=false] `true` to allow selecting multiple tags.
     * @param {Boolean} [config.onlyCustomTags=false] `true` to show only custom tags.
     * @param {Boolean} [config.allowProviders=false] `true` to allow selecting providers.
     * @param {Boolean} [config.allowCreation=false] `true` to allow the creation of new tags from the dialog box.
     * @param {Boolean} [config.onlyTagsWithChildren=false] `true` to prevent selection of tags without child.
     * @param {String} [config.targetType] The type of tags to display. Set to the type of target to display only tags of this type. Set to null to display all tags.
     * @param {String[]} [config.filterTarget] Filter on target type. Set to the type(s) of target to disable tags of this(those) type(s). Set to null to enable all tags.
     * @param {Object} [config.contextualParameters] Optional contextual parameters to be passed to the server requests
     * @param {Function} config.callback The callback function invoked when tags are selected. The callback function will received the following parameters:
     * @param {Function} config.callback.tags The names of selected tags
     */
    open: function (config)
    {
        this.initialConfig = Ext.applyIf(config || {}, this.getDefaultConfig());
        
        this.initialConfig = Ext.apply(this.initialConfig, {
            'cbContextParametersFct' : this._getCbContextParameters
        });
        
        var tree = this.createTree(config);
        var createAction = Ametys.plugins.cms.tag.TagActions.create;
        Ametys.cms.uihelper.ChooseTagHelper.open(
                config,
                tree,
                createAction
        );
    },

    /**
     * Get the default configuration for dialog box
     * @param {Object} config the widget configuration
     * @return {Object} the default configuration
     */
    getDefaultConfig: function(config)
    {
        return {
            title: "{{i18n plugin.cms:PLUGINS_CMS_HELPER_CHOOSETAG_TITLE}}",
            plugin: 'workspaces',
            url: 'tags.json',
            tagModel: "Ametys.plugins.workspaces.project.tag.ProjectTagNode",
            tagsDAO: "org.ametys.plugins.workspaces.tags.ProjectTagsDAO",
            multiple: false,
            onlyCustomTags: false,
            allowProviders: false,
            allowCreation: false,
            onlyTagsWithChildren: false,
            values: []
        }
    },
    
    /**
     * @protected
     * Creates the tags' tree
     * @return {Ametys.plugins.workspaces.project.helper.TagsTreePanel} The panel
     */
    createTree: function ()
    {
        return Ext.create('Ametys.plugins.workspaces.project.helper.TagsTreePanel', {
            width: 320,
            height: 450,

            tagModel: this.initialConfig.tagModel,
            tagsDAO: this.initialConfig.tagsDAO,
            onlyCustomTags: this.initialConfig.onlyCustomTags,
            allowProviders: this.initialConfig.allowProviders,
            allowCreation: this.initialConfig.allowCreation,
            onlyTagsWithChildren: this.initialConfig.onlyTagsWithChildren,
            targetType: this.initialConfig.targetType,
            filterTarget: this.initialConfig.filterTarget,
            contextualParameters: this.initialConfig.contextualParameters,
            values: this.initialConfig.values,
            taggableObjects: this.initialConfig.taggableObjects,
            url: this.initialConfig.url,
            plugin: this.initialConfig.plugin,
            checkMode: this.initialConfig.multiple
        });
    },
    
    /**
     * @protected
     * Get cb context parameter
     * @return {Object} The parameters
     */
    _getCbContextParameters: function ()
    {
        return {};
    }
});
