/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
Projects = {
    
    // the service selector
    SERVICE_SELECTOR: '.ametys-cms-service',
    
    FILTER_SELECTOR: '[data-ametys-projects-role=filters]',
    
    RESULTS_SELECTOR: '[data-ametys-projects-role=results]',
    
    NORESULT_SELECTOR: '[data-ametys-projects-role=no-result]',
    
    RESULTSTATUS_SELECTOR: '[data-ametys-projects-role=results-status]',
    
    // The site uri prefix
    SITE_URI_PREFIX: null,
    // the site name
    SITE_NAME: null,
    // the sitemap language
    LANG: null,
    
    // last update date
    _lastUpdate: null,
    
    // max number of members to display
    _maxMembers: 3,
    
    // jquery selector
    _selector : null,
    
    // is projects currently open ?
    isOpened: false,
    
    /**
     * Configure the projects
     * @param {Number} maxMembers the max number of members to display. Default to 3.
     */
    configure: function(siteUriPrefix, siteName, lang, maxMembers)
    {
        Projects.SITE_URI_PREFIX = siteUriPrefix;
        Projects.SITE_NAME = siteName;
        Projects.LANG = lang;
        
        this._maxMembers = maxMembers || 3;
    },
    
    /**
     * Open and load projects if necessary
     * @param {String} selector a jquery selector
     * @param {Function} [callback] a callback function to call after loading
     * @param {Boolean} [force] set to true to force reload
     */
    open: function(selector, callback, force)
    {
        if (!this._lastUpdate)
        {
            // First initalization: add listeners
            AmetysFront.Event.listen("projects-opened", function(data) {
                Projects.isOpened = true;
	        })
	         
	        AmetysFront.Event.listen("projects-closed", function(data) {
	           Projects.isOpened = false;
	        })
        }
        
        let now = new Date().getTime();
        
        if (!Projects._lastUpdate || force)
        {
            Projects._lastUpdate = now;
            
            let $selector = $j(selector);
            
            AmetysFront.Event.fire("projects-loading");
            
            let data = {
                'maxMembers': this._maxMembers
            };
            
            this._handleFilters($selector, data);
            
            $j.ajax({
                url: Projects.SITE_URI_PREFIX + "/_plugins/workspaces/" + Projects.SITE_NAME + "/" + Projects.LANG + "/projects.xml",
                type: 'POST',
                dataType: 'html',
                data: data,
                traditional: true,
                success: function(data, status, error) 
                {
                    var tmpDoc = document.implementation.createHTMLDocument('');
                    tmpDoc.documentElement.innerHTML = data;
                    
                    var $content = $j(tmpDoc).find('body');
                    
                    var $tmp = $j('<div style="display: none; position: absolute; top: -10000">').appendTo('body');
                    $tmp.html($content.html());
                    
                    var $results = $selector.find(Projects.RESULTS_SELECTOR);
                    // replace
                    $results.html($tmp.find("ul.ametys-projects__list"));
                    
                    var nbVisibleResults = $results.find("ul.ametys-projects__list > li").length;
                    nbVisibleResults == 0 ? $selector.find(Projects.NORESULT_SELECTOR).show() : $selector.find(Projects.NORESULT_SELECTOR).hide();
                    
                    if (force)
                    {
                        // RGAA purpose: update a live zone with role="status" when projects' list is reloaded after a search filter
                        if (nbVisibleResults == 0)
                        {
                            $selector.find(Projects.RESULTSTATUS_SELECTOR).html("{{i18n PLUGINS_WORKSPACES_SERVICE_MY_PROJECTS_STATUS_SEARCH_NORESULT}}");
                        }
                        else
                        {
                            // Number of results displayed after executing filter
                            $selector.find(Projects.RESULTSTATUS_SELECTOR).html(nbVisibleResults + "{{i18n PLUGINS_WORKSPACES_SERVICE_MY_PROJECTS_STATUS_SEARCH_RESULTS}}");
                        }
                    }
                    
                    if (typeof callback == 'function')
                    {
                        callback();
                    }
                    
                    $tmp.remove();
                    $j(tmpDoc).remove();
                    
                    AmetysFront.Event.fire("projects-loaded");
                },
                
                error: function(result, status, error) {
                    console.error("Unable to get projects. Status is " + result.status + ".");
                }
            });
        }
    },
    
    _handleFilters: function($selector, params)
    {
        var $filter = $selector.find(Projects.FILTER_SELECTOR);
        
        $filter.serializeArray()
            .forEach(function(param) {
                var name = param.name,
                val = param.value;
	            if (name && params.hasOwnProperty(name) && val)
	            {
	                let valuesArray = $j.makeArray(params[name]);
	                valuesArray.push(val);
                    params[name] = valuesArray;
	            }
	            else if (name && val)
	            {
	                params[name] = val;
	            }
            });
    },
    
    /**
     * Change the favorite status for a given project
     * @param {String} projectName the project name
     * @param {Boolean} favorite The new favorite status
     * @param {Function} callback the cakkback function to call after chaging favorite status. Can be null.
     */
    setFavorite: function(projectName, favorite, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.workspaces.project.favorites.FavoritesHelper',
            methodName: 'setFavoriteProject',
            parameters: [projectName, favorite],
            callback: {
                handler: function (success) {
                    if (typeof callback == 'function')
                    {
                        callback(success);
                    }
                }
            }
        });
    }
}