/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';
import { useTasksFiltersStore } from '@/stores/tasksFilters';
import { transformToMember } from '@common/helper/MemberHelper.js';
import { useTagsStore } from '@common/store/project-tags/tags'
import { useTasksListStore } from '@/stores/tasksList';

/**
 * Pinia store for tasks
 */
export const useTasksStore = defineStore('tasks',
{
    state: () => (
    {
        tasks: [], // list of taks
    }),
    getters:
    {
        /**
         * Get tasks filtered by the current filters
         */
        getFilteredTasks: state =>
        {
            const tasksFiltersStore = useTasksFiltersStore()
            const tasksListStore = useTasksListStore()
            let tasks = [];
            tasks = state.tasks.filter(task => {
                return filterMyTask(task) && filterTaskByType(task) && filterTaskByText(task);

                function filterMyTask(task)
                {
                    if (tasksFiltersStore.filterByMy)
                    {
                        let currentUserId = window.ametysUser.login + "#" + window.ametysUser.populationId;
                        let usersFiltered = task.assignments.filter(function(u){
                            return u.id === currentUserId;
                        });

                        return usersFiltered.length > 0;
                    }

                    // No filter
                    return true;
                }

                function filterTaskByType(task)
                {
                    if (tasksFiltersStore.filterByType.key == "open")
                    {
                        return task.closeInfo == null;
                    }
                    else if (tasksFiltersStore.filterByType.key == "closed")
                    {
                        return task.closeInfo != null;
                    }

                    // No filter
                    return true;
                }

                function filterTaskByText(task)
                {
                    let taskStringAttributesForSearch = ["label", "description"];

                    if (tasksFiltersStore.filterByName && tasksFiltersStore.filterByName != "" && tasksFiltersStore.filterByName != "#")
                    {
                        let isMatched = false;
                        let text = AmetysFront.Utils.deemphasize(tasksFiltersStore.filterByName.toLowerCase());
                        Object.keys(task).forEach(attr =>
                        {
                            if (taskStringAttributesForSearch.indexOf(attr) !== -1)
                            {
                                if (AmetysFront.Utils.deemphasize(task[attr].toLowerCase()).indexOf(text) !== -1)
                                {
                                    isMatched = true;
                                }
                            }
                            else if (attr === 'tags')
                            {
                                let tagText = text.startsWith("#") ? text.substring(1) : text;
                                task['tags'].forEach(tag =>
                                {
                                    if (AmetysFront.Utils.deemphasize(tag.toLowerCase()).indexOf(tagText) !== -1)
                                    {
                                        isMatched = true;
                                    }
                                });
                            }
                            else if (attr === 'assignments')
                            {
                                task['assignments'].forEach(user =>
                                {
                                    if (AmetysFront.Utils.deemphasize(user.name.toLowerCase()).indexOf(text) !== -1)
                                    {
                                        isMatched = true;
                                    }
                                });
                            }
                            else if (attr === 'checkList')
                            {
                                task['checkList'].forEach(item =>
                                {
                                    if (AmetysFront.Utils.deemphasize(item.label.toLowerCase()).indexOf(text) !== -1)
                                    {
                                        isMatched = true;
                                    }
                                });
                            }
                            else if (attr === 'attachments')
                            {
                                task['attachments'].forEach(file =>
                                {
                                    if (AmetysFront.Utils.deemphasize(file.name.toLowerCase()).indexOf(text) !== -1)
                                    {
                                        isMatched = true;
                                    }
                                });
                            }
                            else if (attr === 'comments')
                            {
                                task['comments'].forEach(comment =>
                                {
                                    if (!comment.isDeleted && AmetysFront.Utils.deemphasize(comment.text.toLowerCase()).indexOf(text) !== -1)
                                    {
                                        isMatched = true;
                                    }
                                    comment.subComments.forEach(subComment =>
                                    {
                                        if (!subComment.isDeleted && AmetysFront.Utils.deemphasize(subComment.text.toLowerCase()).indexOf(text) !== -1)
                                        {
                                            isMatched = true;
                                        }
                                    });
                                });
                            }
                            else if (attr === 'tasksListId')
                            {
                                let sections = tasksListStore.tasksLists.filter(tl => tl.id == task[attr] && AmetysFront.Utils.deemphasize(tl.label.toLowerCase()).indexOf(text) !== -1);
                                if (sections.length)
                                {
                                    isMatched = true;
                                }
                            }
                        });

                        return isMatched;
                    }

                    // No filter
                    return true;
                }
            });

            return tasks;
        }
    },
    actions:
    {
        /**
         * Load all tasks
         */
        async loadTasks()
        {
            AmetysFront.Event.fire('loader', true)

            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'getTasks',
                    parameters: []
                })
                .then(tasks =>
                {
                    for (let task of tasks)
                    {
                        this.transformTask(task);
                    }
                    this.tasks = tasks;
                    AmetysFront.Event.fire('loader', false)
                })
                .catch(error =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASK_ERROR_MSG,
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                })
        },

        /**
         * Add a task
         * @param {Object} task the task to add
         * @param {Array} newFiles the files to add to the task
         * @param {Array} newFileNames the names of the files to add to the task
         */
        async addTask(task, newFiles, newFileNames)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_ADD
            });

            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'addTask',
                    parameters: [
                        task.tasksListId,
                        task,
                        newFiles,
                        newFileNames
                    ]
                })
                .then(data =>
                {
                    this.transformTask(data.task);
                    this.tasks.push(data.task);
                    const tagsStore = useTagsStore();
                    tagsStore.addTags(data.newTags);

                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_ADD
                    });
                })
                .catch(error =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_ERROR_MSG,
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                })
        },

        /**
         * Edit a task
         * @param {Object} modifiedTask the modified task
         * @param {Array} newFiles the new files to add to the task
         * @param {Array} newFileNamesthe names of the new files to add to the task
         * @param {Array} deleteFiles the files to delete from the task
         */
        async editTask(modifiedTask, newFiles, newFileNames, deleteFiles)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_EDIT
            });

            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'editTask',
                    parameters: [
                        modifiedTask.id,
                        modifiedTask,
                        newFiles,
                        newFileNames,
                        deleteFiles
                    ]
                })
                .then(data =>
                {
                    this.transformTask(data.task);
                    this.tasks = this.tasks.map(t => t.id === data.task.id ? data.task : t);
                    const tagsStore = useTagsStore();
                    tagsStore.addTags(data.newTags);

                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_EDIT
                    });
                })
                .catch(error =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_EDIT_TASK_ERROR_MSG,
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                })
        },

        /**
         * Delete a task
         * @param {String} taskId the id of the task to delete
         */
        async deleteTask(taskId)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_DELETE
            });

            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'deleteTask',
                    parameters: [taskId]
                })
                .then(() =>
                {
                    this.tasks = this.tasks.filter(task => task.id != taskId);

                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_DELETE
                    });
                })
                .catch(error =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_DELETE_TASK_ERROR_MSG,
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                })
        },

        /**
         * move the task in a new task list
         * @param {String} tasksListId the id of the current task list
         * @param {String} taskId the id of the task to move
         * @param {Number} newPosition the new position of the task in the task list
         */
        async moveTask(tasksListId, taskId, newPosition)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_MOVE
            });

            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'moveTask',
                    parameters: [tasksListId, taskId, newPosition]
                })
                .then(task =>
                {
                    this.transformTask(task);
                    this.tasks = this.tasks.map(t => t.id === task.id ? task : t);



                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_MOVE
                    });
                })
                .catch(error =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_MOVE_TASK_ERROR_MSG,
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                })
        },

        /**
         * transform the task, use internally in store
         * @param {Object} task the task to transform
         */
        transformTask(task)
        {
            this.transformMembers(task);

            for (let com of task.comments)
            {
                this.transformComment(com);
            }
        },

        /**
         * transform the members of task, use internally in store
         * @param {Object} task the task to transform
         */
        transformMembers(task)
        {
            if (task.assignments)
            {
                task.assignments = task.assignments.map(transformToMember);
            }
        },

        /**
         * transform the comment, use internally in store
         * @param {Object} comment the comment to transform
         */
        transformComment(comment)
        {
            let users = comment.userLikes;
            let currentUserId = window.ametysUser.login + "#" + window.ametysUser.populationId;
            comment.isLiked = users ? users.indexOf(currentUserId) != -1 : false;
            comment.canHandle = comment.author ? comment.author.id === currentUserId : false;
            for (let com of comment.subComments)
            {
                this.transformComment(com);
            }
        },

        /**
         * Update a task in the state
         * @param {Object} task the task to update
         */
        updateTaskInState(task)
        {
            this.tasks = this.tasks.map(t => t.id === task.id ? task : t);
        }
    }

})