/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Calendar Interface
 */
public interface Calendar
{
    /**
     * Enumeration of calendar visiblity
     */
    public enum CalendarVisibility
    {
        /** Public */
        PUBLIC,
        /** Private */
        PRIVATE;
    }
    
    /**
     * Get the id of the calendar
     * @return the id
     */
    public String getId();
    
    /**
     * Get the name of the calendar
     * @return the name
     */
    public String getName();
    
    /**
     * Get the description of the calendar
     * @return the description
     */
    public String getDescription();
    
    /**
     * Get the project
     * @return the project
     */
    public Project getProject();
    
    /**
     * Get the color of the calendar
     * @return the color of the calendar
     */
    public String getColor();
    
    /**
     * Get the calendar visibility
     * @return the calendar visibility
     */
    public CalendarVisibility getVisibility();
    
    /**
     * Get the template description of the calendar
     * @return the template description of the calendar
     */
    public String getTemplateDescription();
    
    /**
     * Get the child calendars
     * @return the child calendars
     */
    public List<Calendar> getChildCalendars();
    
    /**
     * Get all events of the calendar
     * @return the list of events
     */
    public List<CalendarEvent> getAllEvents();
    
    /**
     * Get the events and its occurrences of the calendar between two dates
     * @param startDate Begin date
     * @param endDate End date
     * @return the map of events and its occurrences
     */
    public Map<CalendarEvent, List<CalendarEventOccurrence>> getEvents(ZonedDateTime startDate, ZonedDateTime endDate);
    
    /**
     * Get the token used by the ICS public URL
     * @return The token, or null if no token exists for the calendar
     */
    public String getIcsUrlToken();
}
