/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;

/**
 * Component listing the available colors for a {@link Calendar}
 */
public class CalendarColorsComponent implements Component, Configurable
{
    /** The component role */
    public static final String ROLE = CalendarColorsComponent.class.getName();

    private Map<String, CalendarColor> _colors;
    private Map<String, CalendarColor> _allColors;
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Map<String, CalendarColor> allColors = new LinkedHashMap<>();
        Map<String, CalendarColor> calendarColors = new LinkedHashMap<>();
        
        for (Configuration colorConfiguration : configuration.getChildren("resourceColor"))
        {
            String id = colorConfiguration.getAttribute("id");
            String label = colorConfiguration.getChild("label").getValue();
            String bgColor = colorConfiguration.getChild("bg").getValue();
            String fgColor = colorConfiguration.getChild("fg").getValue();
            
            CalendarColor color = new CalendarColor(id, label, bgColor, fgColor);
            allColors.put(id, color);
        }
        
        for (Configuration colorConfiguration : configuration.getChildren("color"))
        {
            String id = colorConfiguration.getAttribute("id");
            String label = colorConfiguration.getChild("label").getValue();
            String bgColor = colorConfiguration.getChild("bg").getValue();
            String fgColor = colorConfiguration.getChild("fg").getValue();
            
            CalendarColor color = new CalendarColor(id, label, bgColor, fgColor);
            calendarColors.put(id, color);
            allColors.put(id, color);
        }
        
        _colors = calendarColors;
        _allColors = allColors;
    }

    /**
     * Get the colors of calendars
     * @return colors
     */
    public Map<String, CalendarColor> getColors()
    {
        return _colors;
    }
    
    /**
     * Get the colors of calendars including those restricted for resource calendars
     * @return colors
     */
    public Map<String, CalendarColor> getAllColors()
    {
        return _allColors;
    }
    
    /**
     * Get the color by id
     * @param id the id
     * @return the color
     */
    public CalendarColor getColor(String id)
    {
        return _allColors.get(id);
    }
    
    /**
     * Class representing a calendar color
     *
     */
    public class CalendarColor 
    {
        String _bgColor;
        String _fgColor;
        String _label;
        String _id;
        
        /**
         * Constructor
         * @param id the id
         * @param label the label
         * @param bgColor the color
         * @param fgCOlor the text color
         */
        public CalendarColor (String id, String label, String bgColor, String fgCOlor)
        {
            _id = id;
            _label = label;
            _bgColor = bgColor;
            _fgColor = fgCOlor;
        }
        
        /**
         * Get the id
         * @return the id
         */
        public String getId()
        {
            return _id;
        }
        
        /**
         * Get the label
         * @return the label
         */
        public String getLabel()
        {
            return _label;
        }
        
        /**
         * Get the color
         * @return the color
         */
        public String getColor()
        {
            return _bgColor;
        }
        
        /**
         * Get the text color
         * @return the text color
         */
        public String getTextColor()
        {
            return _fgColor;
        }
    }
}
