/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.messagingconnector.MessagingConnector;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventAttendee;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEvent;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Manager for manipulating messaging connector linked to calendars event of a project
 *
 */
public class MessagingConnectorCalendarManager extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = MessagingConnectorCalendarManager.class.getName();
    
    /** Property's name for synchronized id id */
    public static final String PROPERTY_SYNCHRONIZED_ID = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":synchronizedId";
    
    private MessagingConnector _messagingConnector;
    private UserManager _userManager;
    private AmetysObjectResolver _resolver;
    private I18nUtils _i18nUtils;
    private Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        boolean hasService = manager.hasService(MessagingConnector.ROLE);
        _messagingConnector = hasService ? (MessagingConnector) manager.lookup(MessagingConnector.ROLE) : null;
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Set the synchronized id (for messaging connector) to the event
     * @param event the event
     * @param synchronizedId the synchronized id
     */
    public void setSynchronizedId(JCRCalendarEvent event, String synchronizedId)
    {
        try
        {
            event.getNode().setProperty(PROPERTY_SYNCHRONIZED_ID, synchronizedId);
            event.saveChanges();
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred setting synchronized id for event " + event.getId());
        }
    }
    
    /**
     * Get the synchronized id (for messaging connector) of the event
     * @param event the event
     * @return the synchronized id
     */
    public String getSynchronizedId(JCRCalendarEvent event)
    {
        try
        {
            Node eventNode = event.getNode();
            if (eventNode.hasProperty(PROPERTY_SYNCHRONIZED_ID))
            {
                return eventNode.getProperty(PROPERTY_SYNCHRONIZED_ID).getString();
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred getting synchronized id for event " + event.getId());
        }
        
        return null;
    }
    
    /**
     * Set attendees to the event
     * @param eventId the event id
     * @param attendees the list of attendees
     */
    public void setAttendees(String eventId, List<CalendarEventAttendee> attendees)
    {
        if (attendees != null)
        {
            JCRCalendarEvent event = (JCRCalendarEvent) _resolver.resolveById(eventId);
            try
            {
                event.setAttendees(attendees);
            }
            catch (RepositoryException e)
            {
                getLogger().error("An error occurred setting attendees for event " + eventId, e);
            }
        }
    }
    
    /**
     * Set organiser to the event
     * @param eventId the event id
     * @param organiser the organiser
     */
    public void setOrganiser(String eventId, UserIdentity organiser)
    {
        JCRCalendarEvent event = (JCRCalendarEvent) _resolver.resolveById(eventId);
        if (organiser != null)
        {
            event.setOrganiser(organiser);
        }
    }
    
    /**
     * Add event invitation parameters (attendees and organiser)
     * @param parameters the map of parameters
     * @param eventId the event id
     */
    public void addEventInvitation(Map<String, Object> parameters, String eventId)
    {
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> attendeesList = (List<Map<String, Object>>) parameters.get("attendees");
        List<CalendarEventAttendee> attendees = _getAttendeeListFromParameter(attendeesList);
        setAttendees(eventId, attendees);
        
        @SuppressWarnings("unchecked")
        Map<String, Object> organiserMap = (Map<String, Object>) parameters.get("organiser");
        UserIdentity organiser = _getUserFromParameter(organiserMap);
        setOrganiser(eventId, organiser);
        
        if (organiser != null)
        {
            JCRCalendarEvent event = _resolver.resolveById(eventId);
            _createEvent(event, organiser, attendees);
        }
    }
    
    /**
     * Edit event invitation parameters (attendees and organiser)
     * @param parameters the map of parameters
     * @param eventId the event id
     */
    public void editEventInvitation(Map<String, Object> parameters, String eventId)
    {
        JCRCalendarEvent event = _resolver.resolveById(eventId);

        @SuppressWarnings("unchecked")
        List<Map<String, Object>> attendeesList = (List<Map<String, Object>>) parameters.get("attendees");
        List<CalendarEventAttendee> attendees = _getAttendeeListFromParameter(attendeesList);
        try
        {
            if (attendees == null)
            {
                attendees = event.getAttendees();
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred getting attendees from repository for event " + event.getId());
        }
        setAttendees(eventId, attendees);

        @SuppressWarnings("unchecked")
        Map<String, Object> organiserMap = (Map<String, Object>) parameters.get("organiser");
        UserIdentity organiserFromMap = _getUserFromParameter(organiserMap);
        UserIdentity organiser = organiserFromMap != null ? organiserFromMap : event.getOrganiser();
        setOrganiser(eventId, organiser);
        
        if (_messagingConnector != null)
        {
            String synchronizedId = getSynchronizedId(event);
            if (StringUtils.isNotBlank(synchronizedId) && _messagingConnector.isEventExist(synchronizedId, organiser))
            {
                _editEvent(event, organiser, attendees);
            }
            else if (organiser != null)
            {
                _createEvent(event, organiser, attendees);
            }
        }
    }
    
    /**
     * Create event in the messaging connector
     * @param event the event
     * @param organiser the organiser
     * @param attendees the list of attendee
     */
    protected void _createEvent(JCRCalendarEvent event, UserIdentity organiser, List<CalendarEventAttendee> attendees)
    {
        Map<String, Boolean> attendeesForMessagingConnector = new HashMap<>();
        for (CalendarEventAttendee attendee : attendees)
        {
            if (attendee.isExternal())
            {
                attendeesForMessagingConnector.put(attendee.getEmail(), attendee.isMandatory());
            }
            else
            {
                User user = _userManager.getUser(new UserIdentity(attendee.getLogin(), attendee.getPopulationId()));
                attendeesForMessagingConnector.put(user.getEmail(), attendee.isMandatory());
            }
        }
        
        String synchronizedId = _messagingConnector.createEvent(_getEventTitle(event), event.getDescription(), event.getLocation(), event.getFullDay(), event.getStartDate(), event.getEndDate(), event.getRecurrenceType(), event.getRepeatUntil(), attendeesForMessagingConnector, event.getOrganiser());
        setSynchronizedId(event, synchronizedId);
    }
    
    /**
     * Delete event in the messaging connector
     * @param event the event
     */
    public void deleteEvent(JCRCalendarEvent event)
    {
        if (isEventSynchronized(event.getId()))
        {
            String synchronizedId = getSynchronizedId(event);
            _messagingConnector.deleteEvent(synchronizedId, event.getOrganiser());
        }
    }

    /**
     * True if the event is synchronized in the messaging connector
     * @param eventId the event id
     * @return true if the event exist
     */
    public boolean isEventSynchronized(String eventId)
    {
        if (_messagingConnector == null)
        {
            return false;
        }
        JCRCalendarEvent event = _resolver.resolveById(eventId);
        String synchronizedId = getSynchronizedId(event);
        
        if (StringUtils.isBlank(synchronizedId))
        {
            return false;
        }
        
        try
        {
            return _messagingConnector.isEventExist(synchronizedId, event.getOrganiser());
        }
        catch (UnsupportedOperationException e)
        {
            getLogger().error("An error occurred while checking if event " + eventId + " exists", e);
            return false;
        }
    }
    
    /**
     * Edit event in the messaging connector
     * @param event the event
     * @param organiser the organiser
     * @param attendees the list of attendee
     */
    protected void _editEvent(JCRCalendarEvent event, UserIdentity organiser, List<CalendarEventAttendee> attendees)
    {
        String synchronizedId = getSynchronizedId(event);
        Map<String, Boolean> attendeesForMessagingConnector = null;
        if (attendees != null)
        {
            attendeesForMessagingConnector = new HashMap<>();
            for (CalendarEventAttendee attendee : attendees)
            {
                if (attendee.isExternal())
                {
                    attendeesForMessagingConnector.put(attendee.getEmail(), attendee.isMandatory());
                }
                else
                {
                    User user = _userManager.getUser(new UserIdentity(attendee.getLogin(), attendee.getPopulationId()));
                    attendeesForMessagingConnector.put(user.getEmail(), attendee.isMandatory());
                }
            }
        }
        
        _messagingConnector.updateEvent(synchronizedId, _getEventTitle(event), event.getDescription(), event.getLocation(), event.getFullDay(), event.getStartDate(), event.getEndDate(), event.getRecurrenceType(), event.getRepeatUntil(), attendeesForMessagingConnector, event.getOrganiser());
    }
    
    /**
     * Get the computed title of the event
     * @param event the event
     * @return the computed title of the event
     */
    protected String _getEventTitle(JCRCalendarEvent event)
    {
        Calendar calendar = event.getParent();
        Project project = calendar.getProject();
        
        List<String> parameters = new ArrayList<>();
        parameters.add(project.getTitle());
        parameters.add(event.getTitle());
        I18nizableText titleI18n = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_EVENT_MESSAGING_CONNECTOR_TITLE", parameters);
        
        Request request = ContextHelper.getRequest(_context);
        String lang = (String) request.getAttribute("sitemapLanguage");
        
        return _i18nUtils.translate(titleI18n, lang);
    }
    
    
    /**
     * Get user from the user parameter
     * @param userMap the user map from parameters
     * @return the user
     */
    protected UserIdentity _getUserFromParameter(Map<String, Object> userMap)
    {
        if (userMap != null)
        {
            String login = (String) userMap.get("login");
            String populationId = (String) userMap.get("populationId");
            
            return new UserIdentity(login, populationId);
        }
        
        return null;
    }
    
    /**
     * Get attendees list from the attendees parameter
     * @param attendeesList the attendees list from parameters
     * @return the attendees list
     */
    protected List<CalendarEventAttendee> _getAttendeeListFromParameter(List<Map<String, Object>> attendeesList)
    {
        if (attendeesList != null)
        {
            List<CalendarEventAttendee> attendees = new ArrayList<>();
            for (Map<String, Object> attendee : attendeesList)
            {
                CalendarEventAttendee calendarEventAttendee = new CalendarEventAttendee();
                boolean isExternal = (boolean) attendee.get("external");
                if (isExternal)
                {
                    calendarEventAttendee.setEmail((String) attendee.get("email"));
                }
                else
                {
                    calendarEventAttendee.setPopulationId((String) attendee.get("populationId"));
                    calendarEventAttendee.setLogin((String) attendee.get("login"));
                }
                
                calendarEventAttendee.setIsExternal(isExternal);
                calendarEventAttendee.setIsMandatory((boolean) attendee.get("mandatory"));
                
                attendees.add(calendarEventAttendee);
            }
            return attendees;
        }
        
        return null;
    }
    
}
