/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.calendars.resources;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.util.Text;

import org.ametys.core.observation.Event;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.calendars.AbstractCalendarDAO;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.calendars.events.ModifiableCalendarEvent;
import org.ametys.plugins.workspaces.calendars.helper.RecurrentEventHelper;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarResource;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarResourceFactory;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Calendar Resource DAO
 *
 */
public class CalendarResourceDAO extends AbstractCalendarDAO
{

    /** Avalon Role */
    public static final String ROLE = CalendarResourceDAO.class.getName();
        
    /**
     * Delete a calendar
     * @param id The id of the calendar
     * @return The result map with id, parent id and message keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteResource(String id)
    {
        Map<String, Object> result = new HashMap<>();

        assert id != null;
        
        CalendarResource resource = _resolver.resolveById(id);
        ModifiableExplorerNode resourcesRoot = resource.getParent();
        
        _checkUserRights(resourcesRoot, RIGHTS_HANDLE_RESOURCE);
        
        if (!_explorerResourcesDAO.checkLock(resource))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to delete resource'" + resource.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        String parentId = resourcesRoot.getId();
        String name = resource.getName();
        String path = resource.getPath();
        
        resource.remove();
        resourcesRoot.saveChanges();

        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);

        Project project = _workspaceHelper.getProjectFromRequest();
        for (Calendar calendar : calendarModule.getCalendars(project, true))
        {
            _removeResource(calendar, id, calendarModule.getCalendarsRoot(project, true));
        }
      
        Calendar resourceCalendar = calendarModule.getResourceCalendar(project);
        _removeResource(resourceCalendar, id, calendarModule.getResourceCalendarRoot(project, true));
     
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, id);
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parentId);
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, path);
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_RESOURCE_DELETED, _currentUserProvider.getUser(), eventParams));
        
        result.put("id", id);
        result.put("parentId", parentId);
        
        return result;
    }

    private void _removeResource(Calendar calendar, String id, ModifiableResourceCollection modifiableResourceCollection)
    {
        boolean saveChanges = false;
        List<ModifiableCalendarEvent> events = calendar.getAllEvents()
            .stream()
            .filter(ModifiableCalendarEvent.class::isInstance)
            .map(ModifiableCalendarEvent.class::cast)
            .toList();
        
        for (ModifiableCalendarEvent event : events)
        {
            List<String> resourceIds = event.getResources();
            if (resourceIds.contains(id))
            {
                List<String> resourcesWithoutDeletedResource = new ArrayList<>(resourceIds);
                resourcesWithoutDeletedResource.remove(id);
                event.setResources(resourcesWithoutDeletedResource);
                saveChanges = true;
            }
        }
        
        if (saveChanges)
        {
            modifiableResourceCollection.saveChanges();
        }
    }
    
    /**
     * Get the resources from project
     * @return the list of resources
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getResources()
    {
        List<Map<String, Object>> resourcesInfo = new ArrayList<>();
        Project project = _workspaceHelper.getProjectFromRequest();
        
        _checkReadAccess(project, CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        for (CalendarResource resource : getProjectResources(project))
        {
            resourcesInfo.add(getCalendarResourceData(resource));
        }
        
        return resourcesInfo;
    }

    
    /**
     * Get all resources from given projets
     * @param project the project
     * @return All resources as JSON
     */
    public List<CalendarResource> getProjectResources(Project project)
    {
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        ModifiableResourceCollection resourcesRoot = calendarModule.getCalendarResourcesRoot(project, true);
        
        return resourcesRoot.getChildren()
            .stream()
            .filter(CalendarResource.class::isInstance)
            .map(CalendarResource.class::cast)
            .collect(Collectors.toList());
    }

    /**
     * Get calendar info
     * @param calendarResource The calendar
     * @return the calendar data in a map
     */
    public Map<String, Object> getCalendarResourceData(CalendarResource calendarResource)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("id", calendarResource.getId());
        result.put("title", Text.unescapeIllegalJcrChars(calendarResource.getTitle()));
        result.put("icon", calendarResource.getIcon());
        result.put("instructions", calendarResource.getInstructions());
        return result;
    }

    /**
     * Add a calendar
     * @param title The resource title
     * @param icon The resource icon
     * @param instructions The resource instructions
     * @param renameIfExists True to rename if existing
     * @return The result map with id, parentId and name keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addResource(String title, String icon, String instructions, boolean renameIfExists)
    {
        Map<String, Object> result = new HashMap<>();
        
        String originalName = Text.escapeIllegalJcrChars(title);
        
        Project project = _workspaceHelper.getProjectFromRequest();
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        ModifiableTraversableAmetysObject resourcesRoot = calendarModule.getCalendarResourcesRoot(project, true);
        
        _checkUserRights(resourcesRoot, RIGHTS_HANDLE_RESOURCE);
        
        if (BooleanUtils.isNotTrue(renameIfExists) && resourcesRoot.hasChild(originalName))
        {
            getLogger().warn("Cannot create the calendar with name '" + originalName + "', an object with same name already exists.");
            result.put("message", "already-exist");
            return result;
        }
        
        if (!_explorerResourcesDAO.checkLock(resourcesRoot))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify the object '" + resourcesRoot.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        int index = 2;
        String name = originalName;
        while (resourcesRoot.hasChild(name))
        {
            name = originalName + " (" + index + ")";
            index++;
        }
        
        JCRCalendarResource resource = resourcesRoot.createChild(name, JCRCalendarResourceFactory.CALENDAR_RESOURCE_NODETYPE);
        resource.setTitle(title);
        resource.setIcon(icon);
        resource.setInstructions(instructions);
        resourcesRoot.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, resource.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, resourcesRoot.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, title);
        eventParams.put(ObservationConstants.ARGS_PATH, resource.getPath());
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_RESOURCE_CREATED, _currentUserProvider.getUser(), eventParams));
        
        result.put("id", resource.getId());
        result.put("title", Text.unescapeIllegalJcrChars(name));
        result.put("icon", resource.getIcon());
        result.put("instructions", resource.getInstructions());

        return result;
    }

    /**
     * Edit a resource
     * @param id The id of the resource
     * @param title The resource title
     * @param icon The resource icon
     * @param instructions The resource instructions
     * @return The result map with id, parentId and name keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editResource(String id, String title, String icon, String instructions)
    {
        CalendarResource resource = _resolver.resolveById(id);
        
        Map<String, Object> result = new HashMap<>();
        
        String name = Text.escapeIllegalJcrChars(title);
        
        Project project = _workspaceHelper.getProjectFromRequest();
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        ModifiableTraversableAmetysObject resourcesRoot = calendarModule.getCalendarResourcesRoot(project, true);
        
        _checkUserRights(resourcesRoot, RIGHTS_HANDLE_RESOURCE);
                
        if (!_explorerResourcesDAO.checkLock(resourcesRoot))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify the object '" + resourcesRoot.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        resource.setTitle(name);
        resource.setIcon(icon);
        resource.setInstructions(instructions);
        resourcesRoot.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, resource.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, resourcesRoot.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, resource.getPath());
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_RESOURCE_UPDATED, _currentUserProvider.getUser(), eventParams));
        
        result.put("id", resource.getId());
        result.put("title", Text.unescapeIllegalJcrChars(name));
        result.put("icon", resource.getIcon());
        result.put("instructions", resource.getInstructions());

        return result;
    }
    
    /**
     * Get all available resources between two dates for a given id
     * @param eventId the event id
     * @param startDateAsStr The start date.
     * @param endDateAsStr The end date.
     * @param eventStartDateAsStr The start date.
     * @param eventEndDateAsStr The end date.
     * @param recurrenceType The recurrence type.
     * @param isFullDay Is the event full day
     * @param originalOccurrenceStartAsStr original occurrence start date
     * @param zoneIdAsString The zone ID used for the dates
     * @return All available resources as JSON
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> loadResourcesWithAvailability(String eventId, String startDateAsStr, String endDateAsStr, String eventStartDateAsStr, String eventEndDateAsStr, String recurrenceType, boolean isFullDay, String originalOccurrenceStartAsStr, String zoneIdAsString)
    {
        EventRecurrenceTypeEnum recurrenceEnum = EventRecurrenceTypeEnum.valueOf(recurrenceType);
        CalendarEvent event = StringUtils.isNotEmpty(eventId) ? _resolver.resolveById(eventId) : null;

        Project project = _workspaceHelper.getProjectFromRequest();
        
        _checkReadAccess(project, CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        List<Map<String, Object>> resourcesInfo = new ArrayList<>();
        Set<String> collideEventResources = new HashSet<>();
        
        ZonedDateTime startDate = DateUtils.parseZonedDateTime(startDateAsStr);
        ZonedDateTime eventStartDate = DateUtils.parseZonedDateTime(eventStartDateAsStr);
        ZonedDateTime eventEndDate = DateUtils.parseZonedDateTime(eventEndDateAsStr);
        ZonedDateTime originalOccurrenceStartDate = DateUtils.parseZonedDateTime(originalOccurrenceStartAsStr);
        ZonedDateTime endDate = DateUtils.parseZonedDateTime(endDateAsStr);
        long diffInSeconds = ChronoUnit.SECONDS.between(eventStartDate, eventEndDate);

        ZoneId zoneId = ZoneId.of(zoneIdAsString);

        List<ZonedDateTime> occurencesfromDAO = RecurrentEventHelper.getOccurrences(startDate, endDate, eventStartDate, originalOccurrenceStartDate, recurrenceEnum, event != null ? event.getExcludedOccurences()  : new ArrayList<>(), zoneId, endDate);
        
        if (occurencesfromDAO.size() > 0)
        {
            
            ZonedDateTime newStartDate = occurencesfromDAO.get(0);
            ZonedDateTime newEndDate = occurencesfromDAO.get(occurencesfromDAO.size() - 1).plusSeconds(diffInSeconds);
            
            
            List<CalendarEvent> events = _getEvents(newStartDate, newEndDate);
            
            // Check if any other event collide with any occurrence of the event we create/edit
            for (CalendarEvent calendarEvent : events)
            {
                // Don't compute next occurrences for events without resources or the event that is edited
                if (!calendarEvent.getResources().isEmpty() && !calendarEvent.getId().equals(eventId) && _eventCollide(calendarEvent, newStartDate, newEndDate, occurencesfromDAO, diffInSeconds, isFullDay))
                {
                    // Store the resources used by the event that collide with the event we create/edit
                    collideEventResources.addAll(calendarEvent.getResources());
                }
            }
        }

        for (CalendarResource resource : getProjectResources(project))
        {
            Map<String, Object> resourceMap = getCalendarResourceData(resource);
            
            // The resource is available only if it is not used by any of the event that collides with the event we create/edit
            resourceMap.put("available", !collideEventResources.contains(resource.getId()));
            
            resourcesInfo.add(resourceMap);
        }

        return resourcesInfo;
    }
    
    private boolean _eventCollide(CalendarEvent calendarEvent, ZonedDateTime startDate, ZonedDateTime endDate, List<ZonedDateTime> occurencesfromDAO, long diffInSeconds, boolean isFullDay)
    {

        Optional<CalendarEventOccurrence> firstOccurrence = calendarEvent.getFirstOccurrence(isFullDay ? startDate : startDate.truncatedTo(ChronoUnit.DAYS));
        
        if (firstOccurrence.isEmpty())
        {
            return false;
        }
        
        List<ZonedDateTime> excludedOccurences = calendarEvent.getExcludedOccurences();
        
        
        ZonedDateTime firstDateCalendar = firstOccurrence.get().getStartDate();
        
        if (!excludedOccurences.contains(firstDateCalendar) && !_isAvailable(firstOccurrence.get(), occurencesfromDAO, diffInSeconds, isFullDay))
        {
            return true;
        }
        
        Optional<CalendarEventOccurrence> nextOccurrence = calendarEvent.getNextOccurrence(firstOccurrence.get());
        while (nextOccurrence.isPresent() && nextOccurrence.get().before(endDate))
        {

            ZonedDateTime nextDateCalendar = nextOccurrence.get().getStartDate();
            
            if (!excludedOccurences.contains(nextDateCalendar) && !_isAvailable(firstOccurrence.get(), occurencesfromDAO, diffInSeconds, isFullDay))
            {
                return true;
            }
            nextOccurrence = calendarEvent.getNextOccurrence(nextOccurrence.get());
        }
        
        return false;
    }

    /**
     * Get the events between two dates
     * @param startDate Begin date
     * @param endDate End date
     * @return the list of events
     */
    protected List<CalendarEvent> _getEvents(ZonedDateTime startDate, ZonedDateTime endDate)
    {
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        Project project = _workspaceHelper.getProjectFromRequest();

        List<CalendarEvent> eventList = new ArrayList<>();
        for (Calendar calendar : calendarModule.getCalendars(project, true))
        {
            if (calendarModule.canView(calendar))
            {
                for (Map.Entry<CalendarEvent, List<CalendarEventOccurrence>> entry : calendar.getEvents(startDate, endDate).entrySet())
                {
                    CalendarEvent event = entry.getKey();
                    eventList.add(event);
                }
            }
        }
      
        Calendar resourceCalendar = calendarModule.getResourceCalendar(project);
        for (Map.Entry<CalendarEvent, List<CalendarEventOccurrence>> entry : resourceCalendar.getEvents(startDate, endDate).entrySet())
        {
            CalendarEvent event = entry.getKey();
            eventList.add(event);
        }
        
        return eventList; 
    }

    
    private boolean _isAvailable(CalendarEventOccurrence eventOccurrence, List<ZonedDateTime> occurencesfromDAO, long diffInSeconds, boolean isFullDay)
    {
        ZonedDateTime occurrenceStartDate = eventOccurrence.getStartDate();
        ZonedDateTime occurrenceEndDate = eventOccurrence.getEndDate();
        
        
        if (eventOccurrence.isFullDay())
        {                
            occurrenceEndDate = occurrenceEndDate.plusDays(1);
        }
        
        // Compute all occurrence of the event we create/edit
        for (ZonedDateTime occurenceDate : occurencesfromDAO)
        {
            
            ZonedDateTime startDateEvent = occurenceDate;
            ZonedDateTime endDateEvent = occurenceDate.plusSeconds(diffInSeconds);

            if (isFullDay)
            {
               // startDateEvent = startDateEvent.truncatedTo(ChronoUnit.DAYS);
                
               // endDateEvent = endDateEvent.truncatedTo(ChronoUnit.DAYS).plusDays(1);
            }
            if (startDateEvent.isBefore(occurrenceEndDate) && occurrenceStartDate.isBefore(endDateEvent))
            {
                return false;
            }
        }
        
        return true;
    }
    
}
