/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.datapolicy;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.repository.comment.Comment;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.datapolicy.AbstractCommentAndReactionDataPolicy;
import org.ametys.plugins.workspaces.documents.jcr.File;
import org.ametys.plugins.workspaces.documents.jcr.FileFactory;

/**
 * Data policy that anonymize document's comments made by an unknown user
 */
public class DocumentCommentAndReactionDataPolicy extends AbstractCommentAndReactionDataPolicy
{
    @Override
    protected String getLogCategory()
    {
        return "document";
    }
    
    @Override
    protected String getObjectPrimaryType()
    {
        return FileFactory.FILE_NODETYPE;
    }
    
    @Override
    protected boolean handleComment(Node commentNode)
    {
        try
        {
            Pair<Node, String> holderAndCommentId = getObjectNodeAndCommentId(commentNode);
            if (holderAndCommentId != null)
            {
                File file = _resolver.resolve(holderAndCommentId.getLeft(), false);
                Comment comment = file.getComment(holderAndCommentId.getRight());
                
                comment.setAuthor(UserPopulationDAO.UNKNOWN_USER_IDENTITY);
                comment.setAuthorEmail(null);
                comment.setAuthorName(null);
                file.saveChanges();
                
                // No specific observation event  at this point
                
                return true;
            }
        }
        catch (RepositoryException | AmetysRepositoryException e)
        {
            getLogger().error("Failed to retrieve file and comment from comment node '{}'", commentNode,  e);
        }
        
        return false;
    }
}
