/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum;

import java.util.HashMap;
import java.util.Map;

import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.comment.AbstractComment;
import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.observation.Event;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.forum.filters.AcceptedFilter;

/**
 * DAO for manipulating thread comments
 */
public class WorkspaceThreadCommentDAO extends AbstractWorkspaceThreadDAO
{
    /** Avalon Role */
    public static final String ROLE = WorkspaceThreadCommentDAO.class.getName();

    /**
     * Comment a thread
     * @param threadId the thread id
     * @param commentText the comment text
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createComment(String threadId, String commentText)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();

        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_CONTRIBUTE_THREAD);

        RichTextComment comment = createComment(thread, commentText, threadsRoot);

        _setLastContribution(thread, comment);

        _workspaceThreadUserPrefDAO.clearUnreadCommentsNotification(thread.getId(), comment.getId());

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());

        _notifyEvent(thread, comment, ObservationConstants.EVENT_THREAD_COMMENTED);

        threadAsJSON.put("passFilter", true);
        return threadAsJSON;
    }

    /**
     * Edit a thread comment
     * @param threadId the thread id
     * @param commentId the comment Id
     * @param commentText the comment text
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editComment(String threadId, String commentId, String commentText)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();
        
        editComment(thread, commentId, commentText, threadsRoot);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());
        threadAsJSON.put("passFilter", true);
        return threadAsJSON;
    }

    /**
     * Delete a thread comment
     * @param threadId the thread id
     * @param commentId the comment id
     * @param accepted filter by accepted answer
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteComment(String threadId, String commentId, Boolean accepted)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();

        AbstractComment comment = thread.getComment(commentId);
        
        if (!comment.getAuthor().equals(_currentUserProvider.getUser()))
        {
            _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_HANDLE_ALL_THREAD, thread, WorkspaceThreadDAO.RIGHTS_DELETE_OWN_THREAD_ANSWERS);
        }

        deleteComment(thread, commentId, threadsRoot);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());
        AcceptedFilter acceptedFilter = new AcceptedFilter(accepted);

        // We only need to check this filter, as if the other were false the user could'nt access to this thread
        threadAsJSON.put("passFilter", acceptedFilter.passFilter(thread));

        return threadAsJSON;
    }

    /**
     * Answer to a thread comment
     * @param threadId the thread id
     * @param commentId the comment id
     * @param commentText the comment text
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> answerComment(String threadId, String commentId, String commentText)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();

        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_CONTRIBUTE_THREAD);
        
        RichTextComment comment = answerComment(thread, commentId, commentText, threadsRoot);

        _setLastContribution(thread, comment);

        _workspaceThreadUserPrefDAO.clearUnreadCommentsNotification(thread.getId(), comment.getId());

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());
        threadAsJSON.put("passFilter", true);

        _notifyEvent(thread, comment, ObservationConstants.EVENT_THREAD_COMMENTED);

        return threadAsJSON;
    }

    /**
     * Like or unlike a thread's comment
     * @param threadId the thread id
     * @param commentId the comment id
     * @param liked true if the comment is liked, otherwise the comment is unliked
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> likeOrUnlikeComment(String threadId, String commentId, Boolean liked)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();

        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_CONTRIBUTE_THREAD);
        
        likeOrUnlikeComment(thread, commentId, liked, threadsRoot);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());
        threadAsJSON.put("passFilter", true);
        return threadAsJSON;
    }

    private void _setLastContribution(Thread thread, RichTextComment comment)
    {
        thread.setLastContribution(comment.getCreationDate());
        thread.saveChanges();
    }

    /**
     * Accept a thread's comment
     * @param threadId the thread id
     * @param commentId the comment id to set as new accepted answer
     * @param oldCommentId the comment id of the old accepted answer, to set to false if it exists
     * @param accepted filter by accepted answer
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> acceptComment(String threadId, String commentId, String oldCommentId, Boolean accepted)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();
        
        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_HANDLE_ALL_THREAD, thread, WorkspaceThreadDAO.RIGHTS_CREATE_THREAD);

        if (oldCommentId != null)
        {
            _setAcceptedComment(thread, oldCommentId, threadsRoot, false);
        }

        RichTextComment comment = _setAcceptedComment(thread, commentId, threadsRoot, true);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());
        AcceptedFilter acceptedFilter = new AcceptedFilter(accepted);

        _notifyEvent(thread, comment, ObservationConstants.EVENT_THREAD_ACCEPTED);

        // We only need to check this filter, as if the other were false the user could'nt access to this thread
        threadAsJSON.put("passFilter", acceptedFilter.passFilter(thread));
        return threadAsJSON;
    }

    /**
     * Cancel a comment set as accepted answer
     * @param threadId the thread id
     * @param commentId the comment id to remove as accepted answer
     * @param accepted filter by accepted answer
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> cancelAcceptComment(String threadId, String commentId, Boolean accepted)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();
        
        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_HANDLE_ALL_THREAD, thread, WorkspaceThreadDAO.RIGHTS_CREATE_THREAD);

        _setAcceptedComment(thread, commentId, threadsRoot, false);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());
        AcceptedFilter acceptedFilter = new AcceptedFilter(accepted);

        // We only need to check this filter, as if the other were false the user could'nt access to this thread
        threadAsJSON.put("passFilter", acceptedFilter.passFilter(thread));

        return threadAsJSON;
    }

    /**
     * Cancel a comment set as accepted answer
     * @param commentableAmetysObject the commentableAmetysObject
     * @param commentId the comment id to remove as accepted answer
     * @param moduleRoot the module root
     * @param isAccepted true to set the comment as accepted answer.
     * @return The commentableAmetysObject
     */
    protected RichTextComment _setAcceptedComment(CommentableAmetysObject<RichTextComment> commentableAmetysObject, String commentId, ModifiableTraversableAmetysObject moduleRoot, boolean isAccepted)
    {
        RichTextComment comment = commentableAmetysObject.getComment(commentId);

        comment.setAccepted(isAccepted);

        moduleRoot.saveChanges();
        return comment;
    }

    /**
     * Report a thread's comment
     * @param threadId the thread id
     * @param commentId the comment id of the comment to report
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> reportComment(String threadId, String commentId)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();

        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_CONTRIBUTE_THREAD);

        RichTextComment comment = thread.getComment(commentId);

        comment.addReport();

        threadsRoot.saveChanges();

        _notifyEvent(thread, comment, ObservationConstants.EVENT_THREAD_REPORTED);

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());

        threadAsJSON.put("passFilter", true);

        return threadAsJSON;
    }

    /**
     * Clear reports of a thread's comment
     * @param threadId the thread id
     * @param commentId the comment id of the comment to clear
     * @return The thread data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> clearReports(String threadId, String commentId)
    {
        Thread thread = _resolver.resolveById(threadId);

        ModifiableTraversableAmetysObject threadsRoot = thread.getParent();
        
        _checkUserRights(threadsRoot, WorkspaceThreadDAO.RIGHTS_REPORT_NOTIFICATION, thread, WorkspaceThreadDAO.RIGHTS_REPORT_NOTIFICATION_OWN_THREAD);

        RichTextComment comment = thread.getComment(commentId);

        comment.clearReports();

        threadsRoot.saveChanges();

        Map<String, Object> threadAsJSON = _threadJSONHelper.threadAsJSON(thread, getSitemapLanguage(), getSiteName());

        threadAsJSON.put("passFilter", true);

        return threadAsJSON;
    }

    private void _notifyEvent(Thread thread, RichTextComment comment, String eventId)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_THREAD, thread);

        eventParams.put(ObservationConstants.ARGS_THREAD_COMMENT, comment);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, thread.getId());
        
        _observationManager.notify(new Event(eventId, _currentUserProvider.getUser(), eventParams));
    }
}
