/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum.jcr;

import javax.jcr.Node;

import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.DefaultAmetysObjectFactory;
import org.ametys.plugins.workspaces.data.type.ModelItemTypeExtensionPoint;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.plugins.workspaces.forum.Thread;

/**
 * {@link AmetysObjectFactory} for handling {@link JCRThread}.
 */
public class JCRThreadFactory extends DefaultAmetysObjectFactory
{
    /** JCR nodetype for threads */
    public static final String THREAD_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":forum-thread";
    
    /** The thread list model */
    private Model _threadModel;
    
    @Override
    public JCRThread getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new JCRThread(node, parentPath, this);
    }

    /**
     * Get the thread model
     * @return the thread model
     */
    public Model getThreadModel()
    {
        if (_threadModel == null)
        {
            try
            {
                String role = ModelItemTypeExtensionPoint.ROLE_THREAD;
                _threadModel = Model.of(
                        "thread.model.id", 
                        "thread.model.family.id",
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_TITLE, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_CONTENT, false, org.ametys.cms.data.type.ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_AUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_CREATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_LASTMODIFIED, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_CATEGORY, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_CLOSEDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_CLOSEAUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_ATTACHMENTS, true, org.ametys.cms.data.type.ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID, role),
                        DefaultElementDefinition.of(Thread.ATTRIBUTE_LASTCONTRIBUTION, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role)
                        );
            }
            catch (Exception e) 
            {
                getLogger().error("An error occurred getting the thread model", e);
                throw new RuntimeException("An error occurred getting the thread model", e);
            }
        }
        return _threadModel;
    }
}
