/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.members;

import javax.jcr.Node;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelLessComposite;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelLessComposite;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;

/**
 * Class representing of project's member, backed by a JCR node.
 */
public class JCRProjectMember extends DefaultAmetysObject<JCRProjectMemberFactory> implements ModifiableModelLessDataAwareAmetysObject
{
    /** metadata name for the user identity */
    public static final String METADATA_USER = "user";
    /** metadata name for the group identity composite */
    public static final String METADATA_GROUP = "group";
    /** metadata name for the group directory property */
    public static final String METADATA_GROUP_DIRECTORY = "groupDirectory";
    /** metadata name for the group id property */
    public static final String METADATA_GROUP_ID = "groupId";
    private static final String METADATA_TYPE = "type";
    private static final String METADATA_ROLE = "role";
    
    /** The type of the member */
    public enum MemberType
    {
        /** The member is a user */
        USER,
        /** The member is a group */
        GROUP
    }
    
    /**
     * Creates a {@link JCRProjectMember}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the ProjectUserFactory which created the AmetysObject
     */
    public JCRProjectMember(Node node, String parentPath, JCRProjectMemberFactory factory)
    {
        super(node, parentPath, factory);
    }

    public ModifiableModelLessDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getProjectDataTypeExtensionPoint(), repositoryData);
    }

    /**
     * Get the type of the member. It can be a user or a group
     * @return The type of the member
     */
    public MemberType getType()
    {
        String value = getValue(METADATA_TYPE, MemberType.USER.name());
        return MemberType.valueOf(value.toUpperCase());
    }
    
    /**
     * Set the type of the member. It can be a user or a group
     * @param type The type of the member
     */
    public void setType(MemberType type)
    {
        this.setValue(METADATA_TYPE, type.name().toLowerCase());
    }

    /**
     * Get the user identity of the project member
     * @return The user identity
     */
    public UserIdentity getUser()
    {
        return this.getValue(METADATA_USER);
    }
    
    /**
     * Set the user identity
     * @param userIdentity The user identity
     */
    public void setUser(UserIdentity userIdentity)
    {
        this.setValue(METADATA_USER, userIdentity);
    }
    
    /**
     * Get the group identity of the project member
     * @return The group identity
     */
    public GroupIdentity getGroup()
    {
        ModelLessComposite group = this.getComposite(METADATA_GROUP);
        if (group != null)
        {
            return new GroupIdentity(group.getValue(METADATA_GROUP_ID), group.getValue(METADATA_GROUP_DIRECTORY));
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Set the group identity
     * @param groupIdentity The group identity
     */
    public void setGroup(GroupIdentity groupIdentity)
    {
        ModifiableModelLessComposite group = getComposite(METADATA_GROUP, true);
        group.setValue(METADATA_GROUP_DIRECTORY, groupIdentity.getDirectoryId());
        group.setValue(METADATA_GROUP_ID, groupIdentity.getId());
    }
    
    /**
     * Get the role of this project member
     * @return The role
     */
    public String getRole()
    {
        return this.getValue(METADATA_ROLE);
    }
    
    /**
     * Set the role of this project member. Can be null
     * @param role The new role
     */
    public void setRole(String role)
    {
        this.setValue(METADATA_ROLE, role);
    }
    
}
