/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.annotation.Obsolete;

import org.ametys.cms.languages.Language;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.URIResolverExtensionPoint;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.group.Group;
import org.ametys.core.group.GroupDirectoryContextHelper;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.page.UserDirectoryPageResolver;
import org.ametys.plugins.userdirectory.page.UserPage;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.forum.ForumWorkspaceModule;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.objects.Project.InscriptionStatus;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.population.PopulationContextHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.usermanagement.UserManagementException;
import org.ametys.web.usermanagement.UserSignupManager;

/**
 * Helper component for managing project's users
 */
public class ProjectMemberManager extends AbstractLogEnabled implements Serviceable, Component, Contextualizable, Initializable, AsyncObserver, Disposable
{
    /** Avalon Role */
    public static final String ROLE = ProjectMemberManager.class.getName();
    
    /** The id of the members service */
    public static final String __WORKSPACES_SERVICE_MEMBERS = "org.ametys.plugins.workspaces.module.Members";
    
    private static final String __PROJECT_MEMBER_CACHE = "projectMemberCache";
    
    @Obsolete // For v1 project only
    private static final String __PROJECT_RIGHT_PROFILE = "PROJECT";
    
    /** Constants for users project node */
    private static final String __PROJECT_MEMBERS_NODE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":members";
    
    /** The type of the project users node type */
    private static final String __PROJECT_MEMBERS_NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":unstructured";

    /** The type of a project user node type */
    private static final String __PROJECT_MEMBER_NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":project-member";

    private static Pattern __MAIL_BETWEEN_BRACKETS_PATTERN = Pattern.compile("^[^<]*<(.*@.*)>$");
    
    /** Avalon context */
    protected Context _context;
    
    /** Project manager */
    protected ProjectManager _projectManager;

    /** Project rights helper */
    protected ProjectRightHelper _projectRightHelper;

    /** Profiles right manager */
    protected RightProfilesDAO _rightProfilesDAO;

    /** Profile assignment storage */
    protected ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageExtensionPoint;
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    /** Rights manager */
    protected RightManager _rightManager;

    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** Users manager */
    protected UserManager _userManager;

    /** The observation manager */
    protected ObservationManager _observationManager;

    /** Module managers EP */
    protected WorkspaceModuleExtensionPoint _moduleManagerEP;
    
    /** The user helper */
    protected UserHelper _userHelper;

    /** The groups manager */
    protected GroupManager _groupManager;

    /** The population context helper */
    protected PopulationContextHelper _populationContextHelper;

    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    
    /** The project invitation helper */
    protected ProjectInvitationHelper _projectInvitationHelper;

    /** The language manager */
    protected LanguagesManager _languagesManager;
        
    /** The resolver for user directory pages */
    protected UserDirectoryPageResolver _userDirectoryPageResolver;

    /** The page URI resolver. */
    protected URIResolverExtensionPoint _uriResolver;

    /** The group directory context helper */
    protected GroupDirectoryContextHelper _groupDirectoryContextHelper;

    /** The cache manager */
    protected AbstractCacheManager _abstractCacheManager;
    
    /** The user signup manager */
    protected UserSignupManager _userSignupManager;

    /** The project catalogue manager component */
    protected ProjectsCatalogueManager _projectsCatalogueManager;

    /** The helper for project rights */
    protected ProjectRightHelper _projectRightsHelper;

    /** Workspace helper */
    protected WorkspacesHelper _workspaceHelper;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _abstractCacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectRightHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _rightProfilesDAO = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
        _profileAssignmentStorageExtensionPoint = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _populationContextHelper = (PopulationContextHelper) manager.lookup(org.ametys.core.user.population.PopulationContextHelper.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
        _projectInvitationHelper = (ProjectInvitationHelper) manager.lookup(ProjectInvitationHelper.ROLE);
        _languagesManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
        _userDirectoryPageResolver = (UserDirectoryPageResolver) manager.lookup(UserDirectoryPageResolver.ROLE);
        _uriResolver = (URIResolverExtensionPoint) manager.lookup(URIResolverExtensionPoint.ROLE);
        _groupDirectoryContextHelper = (GroupDirectoryContextHelper) manager.lookup(GroupDirectoryContextHelper.ROLE);
        _userSignupManager = (UserSignupManager) manager.lookup(UserSignupManager.ROLE);
        _projectsCatalogueManager = (ProjectsCatalogueManager) manager.lookup(ProjectsCatalogueManager.ROLE);
        _projectRightsHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _abstractCacheManager.createMemoryCache(__PROJECT_MEMBER_CACHE,
                new I18nizableText("plugin.workspaces", "PLUGIN_WORKSPACES_CACHE_PROJECT_MEMBER_LABEL"),
                new I18nizableText("plugin.workspaces", "PLUGIN_WORKSPACES_CACHE_PROJECT_MEMBER_DESCRIPTION"),
                true,
                null);
        
        _observationManager.registerObserver(this);
    }
    
    public void dispose()
    {
        _observationManager.unregisterObserver(this);
    }
    
    /**
     * Retrieve the data of a member of a project, or the default data if no user is provided
     * @param projectName The name of the project
     * @param identity The user or group identity. If null, return the default profiles for a new user
     * @param type The type of the identity. Can be "user" or "group"
     * @return The map of profiles per module for the user
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getProjectMemberData(String projectName, String identity, String type)
    {

        Project project = _projectManager.getProject(projectName);
        
        if (!_projectRightHelper.canEditMember(project))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to get member's rights without convenient right [" + projectName + ", " + identity + "]");
        }
        
        Map<String, Object> result = new HashMap<>();
        
        boolean isTypeUser = JCRProjectMember.MemberType.USER.name().equals(type.toUpperCase());
        boolean isTypeGroup = JCRProjectMember.MemberType.GROUP.name().equals(type.toUpperCase());
        UserIdentity user = Optional.ofNullable(identity)
                                    .filter(id -> id != null && isTypeUser)
                                    .map(UserIdentity::stringToUserIdentity)
                                    .orElse(null);
        GroupIdentity group = Optional.ofNullable(identity)
                                      .filter(id -> id != null && isTypeGroup)
                                      .map(GroupIdentity::stringToGroupIdentity)
                                      .orElse(null);
        
        if (identity != null)
        {
            if (isTypeGroup && group == null)
            {
                result.put("message", "unknown-group");
                result.put("success", false);
                return result;
            }
            else if (isTypeUser && user == null)
            {
                result.put("message", "unknown-user");
                result.put("success", false);
                return result;
            }
        }
        
        
        boolean newMember = true;
        Map<String, String> userProfiles;
        
        if (user != null || group != null)
        {
            JCRProjectMember projectMember = user != null ? _getOrCreateJCRProjectMember(project, user) : _getOrCreateJCRProjectMember(project, group);
            
            newMember = projectMember.needsSave();
            
            String role = projectMember.getRole();
            if (role != null)
            {
                result.put("role", role);
            }
            
            userProfiles = _getMemberProfiles(projectMember, project);
        }
        else
        {
            userProfiles = new HashMap<>();
        }
        
        result.put("profiles", userProfiles);
        result.put("status", newMember ? "new" : "edit");
        result.put("success", true);
        
        return result;
    }

    
    /**
     * Get right profile of a member
     * @param member The member
     * @param project The project name
     * @return a map of the right profile
     */
    private Map<String, String> _getMemberProfiles(JCRProjectMember member, Project project)
    {
        Map<String, String> userProfiles = new HashMap<>();
        
        // Get allowed profile on modules (among the project members's profiles)
        for (WorkspaceModule module : _projectManager.getModules(project))
        {
            String allowedProfileOnProject = _getAllowedProfileOnModule(project, module, member);
            userProfiles.put(module.getId(), allowedProfileOnProject);
        }
        
        return userProfiles;
    }
    
    private String _getAllowedProfileOnModule (Project project, WorkspaceModule module, JCRProjectMember member)
    {
        Set<String> profileIds = _projectRightHelper.getProfilesIds();
        
        AmetysObject moduleObject = module.getModuleRoot(project, false);
        Set<String> allowedProfilesForMember = _getAllowedProfile(member, moduleObject);
        
        for (String allowedProfile : allowedProfilesForMember)
        {
            if (profileIds.contains(allowedProfile))
            {
                // Get the first allowed profile among the project's members profiles
                return allowedProfile;
            }
        }
        
        return null;
    }
    
    /**
     * Add new members and invitation by email
     * @param projectName The project name
     * @param newMembers The members to add (users or groups)
     * @param invitEmails The invitation emails
     * @return the result with errors
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addMembers(String projectName, List<Map<String, String>> newMembers, List<String> invitEmails)
    {
        Project project = _projectManager.getProject(projectName);
        
        if (!_projectRightHelper.canAddMember(project))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried perform operation without convenient right");
        }
        
        Map<String, Object> result = new HashMap<>();
        
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request);
        
        boolean hasError = false;
        boolean inviteError = false;
        boolean unknownProject = false;
        List<String> unknownGroups = new ArrayList<>();
        List<String> unknownUsers = new ArrayList<>();
        List<Map<String, Object>> existingUsers = new ArrayList<>();
        List<Map<String, Object>> membersAdded = new ArrayList<>();
        
        List<String> filteredInvitEmails = new ArrayList<>();
        if (invitEmails != null)
        {
            try
            {
                for (String invitEmail : invitEmails)
                {
                    
                    
                    String filteredInvitEmail = invitEmail;

                    // Regexp pattern to extract "email@domain.com" from "FirstName LastName <email@domain.com>"
                    // ^[^<]*<(.*@.*)>$
                    // ^  => asserts position at start of a line
                    // [^<]* => match any characters that are not '<', so the matched group start at the first bracket
                    // <(.*@.*)> => match text between brackets, containing '@'
                    Matcher matcher =  __MAIL_BETWEEN_BRACKETS_PATTERN.matcher(invitEmail);
                    if (matcher.matches() && matcher.groupCount() == 1)
                    {
                        filteredInvitEmail = matcher.group(1);
                    }
                    
                    Optional<User> userIfExists = _userSignupManager.getUserIfHeExists(filteredInvitEmail, siteName);
                    if (userIfExists.isPresent())
                    {
                        newMembers.add(Map.of(
                            "id", UserIdentity.userIdentityToString(userIfExists.get().getIdentity()),
                            "type", "user"
                        ));
                    }
                    else
                    {
                        filteredInvitEmails.add(filteredInvitEmail);
                    }
                }
            }
            catch (UserManagementException e)
            {
                hasError = true;
                inviteError = true;
                getLogger().error("Impossible to send email invitations", e);
            }
            catch (NotUniqueUserException e)
            {
                hasError = true;
                inviteError = true;
                getLogger().error("Impossible to send email invitations, some user already exist", e);
            }
        }
        
        for (Map<String, String> newMember : newMembers)
        {
            Map<String, Object> addResult = addMember(projectName, newMember.get("id"), newMember.get("type"));
            boolean success = (boolean) addResult.get("success");
            if (!success)
            {
                String error = (String) addResult.get("message");
                if ("unknown-user".equals(error))
                {
                    hasError = true;
                    unknownUsers.add(newMember.get("id"));
                }
                else if ("unknown-group".equals(error))
                {
                    hasError = true;
                    unknownGroups.add(newMember.get("id"));
                }
                else if ("unknown-project".equals(error))
                {
                    hasError = true;
                    unknownProject = true;
                }
                else if ("existing-user".equals(error))
                {
                    existingUsers.add((Map<String, Object>) addResult.get("existing-user"));
                }
            }
            else
            {
                membersAdded.add((Map<String, Object>) addResult.get("member"));
            }
        }
        
        if (!filteredInvitEmails.isEmpty())
        {
            Map<String, String> newProfiles = _getDefaultProfilesByModule();
            
            try
            {
                Map<String, Object> inviteEmails = _projectInvitationHelper.inviteEmails(projectName, filteredInvitEmails, newProfiles);
                List<String> errors = (List<String>) inviteEmails.get("email-error");
                if (!errors.isEmpty())
                {
                    hasError = true;
                    inviteError = true;
                }
                existingUsers.addAll((List<Map<String, Object>>) inviteEmails.get("existing-users"));
            }
            catch (UserManagementException e)
            {
                hasError = true;
                inviteError = true;
                getLogger().error("Impossible to send email invitations", e);
            }
            catch (NotUniqueUserException e)
            {
                hasError = true;
                inviteError = true;
                getLogger().error("Impossible to send email invitations, some user already exist", e);
            }
            
        }
        
        result.put("invite-error", inviteError);
        result.put("existing-users", existingUsers);
        result.put("unknown-groups", unknownGroups);
        result.put("unknown-users", unknownUsers);
        result.put("unknown-project", unknownProject);
        result.put("members-added", membersAdded);
        result.put("success", !hasError);
        
        return result;
    }
    
    /**
     * Add a new member
     * @param projectName The project name
     * @param identity The user or group identity.
     * @param type The type of the identity. Can be "user" or "group"
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addMember(String projectName, String identity, String type)
    {
        Map<String, String> newProfiles = _getDefaultProfilesByModule();
        
        return _setProjectMemberData(projectName, identity, type, newProfiles, null, true);
    }
    
    /**
     * Get a map of each available module, with the default profile
     * @return A map with moduleId : profileId
     */
    protected Map<String, String> _getDefaultProfilesByModule()
    {
        Map<String, String> newProfiles = new HashMap<>();
        
        String defaultProfile = StringUtils.defaultString(Config.getInstance().getValue("workspaces.profile.default"));
        for (String moduleId : _moduleManagerEP.getExtensionsIds())
        {
            newProfiles.put(moduleId, defaultProfile);
        }
        
        return newProfiles;
    }
    
    /**
     * Set the user data in the project
     * @param projectName The project name
     * @param identity The user or group identity.
     * @param type The type of the identity. Can be "user" or "group"
     * @param newProfiles The profiles to affect, mapped by module
     * @param role The user role inside the project
     * @return The result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setProjectMemberData(String projectName, String identity, String type, Map<String, String> newProfiles, String role)
    {
        return _setProjectMemberData(projectName, identity, type, newProfiles, role, false);
    }
    
    /**
     * Set the user data in the project
     * @param projectName The project name
     * @param identity The user or group identity.
     * @param type The type of the identity. Can be "user" or "group"
     * @param newProfiles The profiles to affect, mapped by module
     * @param role The user role inside the project
     * @param isNewUser <code>true</code> if the user is just added
     * @return The result
     */
    protected Map<String, Object> _setProjectMemberData(String projectName, String identity, String type, Map<String, String> newProfiles, String role, boolean isNewUser)
    {
        Map<String, Object> result = new HashMap<>();
        Project project = _projectManager.getProject(projectName);
        
        if (isNewUser && !_projectRightHelper.canAddMember(project) || !isNewUser && !_projectRightHelper.canEditMember(project))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to set member rights without convenient right [" + projectName + ", " + identity + "]");
        }
        
        boolean isTypeUser = JCRProjectMember.MemberType.USER.name().equals(type.toUpperCase());
        boolean isTypeGroup = JCRProjectMember.MemberType.GROUP.name().equals(type.toUpperCase());
        UserIdentity user = Optional.ofNullable(identity)
                                    .filter(id -> id != null && isTypeUser)
                                    .map(UserIdentity::stringToUserIdentity)
                                    .orElse(null);
        GroupIdentity group = Optional.ofNullable(identity)
                                      .filter(id -> id != null && isTypeGroup)
                                      .map(GroupIdentity::stringToGroupIdentity)
                                      .orElse(null);
        
        if (group == null && user == null)
        {
            result.put("success", false);
            result.put("message", isTypeGroup ? "unknown-group" : "unknown-user");
            return result;
        }
        if (isNewUser && isTypeUser && _getProjectMember(project, user) != null)
        {
            result.put("success", false);
            result.put("message", "existing-user");
            result.put("existing-user", _userHelper.user2json(user, true));
            
            return result;
        }
        
        JCRProjectMember projectMember  = isTypeUser ? addOrUpdateProjectMember(project, user, newProfiles) : addOrUpdateProjectMember(project, group, newProfiles);
        if (projectMember != null)
        {
            Request request = ContextHelper.getRequest(_context);
            String lang = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
            
            ProjectMember member = isTypeUser ? new ProjectMember(_userManager.getUser(user), projectMember.getRole(), false) : new ProjectMember(_groupManager.getGroup(group));
            result.put("member", _member2Json(member, lang));
        }
        
        result.put("success", projectMember != null);
        return result;
    }
    
    /**
     * Add a user to a project with open inscriptions, using the default values
     * @param project The project
     * @param user The user
     * @return the added member in case of success, null otherwise
     */
    public JCRProjectMember addProjectMember(Project project, UserIdentity user)
    {
        InscriptionStatus inscriptionStatus = project.getInscriptionStatus();
        if (!inscriptionStatus.equals(InscriptionStatus.OPEN))
        {
            return null;
        }
        
        return addOrUpdateProjectMember(project, user, Map.of());
    }
    
    /**
     * Add a user to a project, using the provided profile values
     * @param project The project
     * @param user The user
     * @param allowedProfiles the profile values
     * @return the added member in case of success, null otherwise
     */
    public JCRProjectMember addOrUpdateProjectMember(Project project, UserIdentity user, Map<String, String> allowedProfiles)
    {
        return addOrUpdateProjectMember(project, user, allowedProfiles, _currentUserProvider.getUser());
    }
    /**
     * Add a user to a project, using the provided profile values
     * @param project The project
     * @param user The user
     * @param allowedProfiles the profile values
     * @param issuer identity of the user that approved the member
     * @return the added member in case of success, null otherwise
     */
    public JCRProjectMember addOrUpdateProjectMember(Project project, UserIdentity user, Map<String, String> allowedProfiles, UserIdentity issuer)
    {
        if (user == null)
        {
            return null;
        }
        
        JCRProjectMember projectMember = _getOrCreateJCRProjectMember(project, user);
        _setMemberProfiles(allowedProfiles, projectMember, project);
        _saveAndNotifyProjectMemberUpdate(project, projectMember, UserIdentity.userIdentityToString(user), issuer);
        return projectMember;
    }
    
    /**
     * Add a group to a project, using the provided profile values
     * @param project The project
     * @param group The group
     * @param allowedProfiles the profile values
     * @return the added member in case of success, null otherwise
     */
    public JCRProjectMember addOrUpdateProjectMember(Project project, GroupIdentity group, Map<String, String> allowedProfiles)
    {
        if (group == null)
        {
            return null;
        }
        
        JCRProjectMember projectMember = _getOrCreateJCRProjectMember(project, group);
        _setMemberProfiles(allowedProfiles, projectMember, project);
        _saveAndNotifyProjectMemberUpdate(project, projectMember, GroupIdentity.groupIdentityToString(group), _currentUserProvider.getUser());
        return projectMember;
    }

    private void _saveAndNotifyProjectMemberUpdate(Project project, JCRProjectMember projectMember, String userIdentityString, UserIdentity issuer)
    {
        project.saveChanges();
        
        _getCache().invalidate(ProjectMemberCacheKey.of(project.getId(), null));
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_MEMBER, projectMember);
        eventParams.put(ObservationConstants.ARGS_MEMBER_ID, projectMember.getId());
        eventParams.put(ObservationConstants.ARGS_PROJECT, project);
        eventParams.put(ObservationConstants.ARGS_PROJECT_ID, project.getId());
        eventParams.put(ObservationConstants.ARGS_MEMBER_IDENTITY, userIdentityString);
        eventParams.put(ObservationConstants.ARGS_MEMBER_IDENTITY_TYPE, projectMember.getType());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_MEMBER_ADDED, issuer, eventParams));
    }

    /**
     * Set the profiles for a member
     * @param newProfiles The allowed profile by module
     * @param projectMember The member
     * @param project The project
     */
    private void _setMemberProfiles(Map<String, String> newProfiles, JCRProjectMember projectMember, Project project)
    {
        String defaultProfile = project.getDefaultProfile();
        Set<String> defaultProfiles;
        if (StringUtils.isEmpty(defaultProfile))
        {
            defaultProfiles = Set.of();
        }
        else
        {
            defaultProfiles = Set.of(defaultProfile);
        }
        
        for (WorkspaceModule module : _moduleManagerEP.getModules())
        {
            Set<String> moduleProfiles;
            if (newProfiles.containsKey(module.getId()))
            {
                String profile = newProfiles.get(module.getId());
                moduleProfiles = StringUtils.isEmpty(profile) ? Set.of() : Set.of(profile);
            }
            else
            {
                moduleProfiles = defaultProfiles;
            }
            setProfileOnModule(projectMember, project, module, moduleProfiles);
        }
    }
    
    /**
     * Affect profiles for a member on a given module
     * @param member The member
     * @param project The project
     * @param module The module
     * @param allowedProfiles The allowed profiles for the module
     */
    public void setProfileOnModule(JCRProjectMember member, Project project, WorkspaceModule module, Set<String> allowedProfiles)
    {
        if (module != null && _projectManager.isModuleActivated(project, module.getId()))
        {
            AmetysObject moduleObject = module.getModuleRoot(project, false);
            _setMemberProfiles(member, allowedProfiles, moduleObject);
        }
    }
    
    private Set<String> _getAllowedProfile(JCRProjectMember member, AmetysObject object)
    {
        if (MemberType.GROUP == member.getType())
        {
            Map<GroupIdentity, Map<UserOrGroup, Set<String>>> profilesForGroups = _profileAssignmentStorageExtensionPoint.getProfilesForGroups(object, Set.of(member.getGroup()));
            return Optional.ofNullable(profilesForGroups.get(member.getGroup())).map(a -> a.get(UserOrGroup.ALLOWED)).orElse(Set.of());
        }
        else
        {
            Map<UserIdentity, Map<UserOrGroup, Set<String>>> profilesForUsers = _profileAssignmentStorageExtensionPoint.getProfilesForUsers(object, member.getUser());
            return Optional.ofNullable(profilesForUsers.get(member.getUser())).map(a -> a.get(UserOrGroup.ALLOWED)).orElse(Set.of());
        }
    }

    private void _setMemberProfiles(JCRProjectMember member, Set<String> allowedProfiles, AmetysObject object)
    {
        Set<String> currentAllowedProfiles = _getAllowedProfile(member, object);
        
        Collection<String> profilesToRemove  = CollectionUtils.removeAll(currentAllowedProfiles, allowedProfiles);
        
        Collection<String> profilesToAdd  = CollectionUtils.removeAll(allowedProfiles, currentAllowedProfiles);
        
        for (String profileId : profilesToRemove)
        {
            _removeProfile(member, profileId, object);
        }
        
        for (String profileId : profilesToAdd)
        {
            _addProfile(member, profileId, object);
        }
        
        Collection<String> updatedProfiles = CollectionUtils.union(profilesToAdd, profilesToRemove);
        
        if (updatedProfiles.size() > 0)
        {
            _notifyAclUpdated(_currentUserProvider.getUser(), object, updatedProfiles);
        }
    }
    
    private void _removeProfile(JCRProjectMember member, String profileId, AmetysObject aclObject)
    {
        if (MemberType.GROUP == member.getType())
        {
            _profileAssignmentStorageExtensionPoint.removeAllowedProfileFromGroup(member.getGroup(), profileId, aclObject);
        }
        else
        {
            _profileAssignmentStorageExtensionPoint.removeAllowedProfileFromUser(member.getUser(), profileId, aclObject);
        }
    }
    
    private void _addProfile(JCRProjectMember member, String profileId, AmetysObject aclObject)
    {
        if (MemberType.GROUP == member.getType())
        {
            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(member.getGroup(), profileId, aclObject);
        }
        else
        {
            _profileAssignmentStorageExtensionPoint.allowProfileToUser(member.getUser(), profileId, aclObject);
        }
    }

    private void _removeMemberProfiles(JCRProjectMember member, AmetysObject object)
    {
        Set<String> currentAllowedProfiles = _getAllowedProfile(member, object);
        
        for (String allowedProfile : currentAllowedProfiles)
        {
            _removeProfile(member, allowedProfile, object);
        }
        
        if (currentAllowedProfiles.size() > 0)
        {
            ((ModifiableAmetysObject) object).saveChanges();
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, object);
            eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT_IDENTIFIER, object.getId());
            eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, currentAllowedProfiles);
            eventParams.put(org.ametys.cms.ObservationConstants.ARGS_ACL_SOLR_CACHE_UNINFLUENTIAL, true);
            
            _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, _currentUserProvider.getUser(), eventParams));
        }
    }
    
    /**
     * Get the members of a project, sorted by managers, non empty role and name
     * @param projectName the project's name
     * @param lang the language to get user content
     * @return the members of project
     * @throws IllegalAccessException if an error occurred
     * @throws AmetysRepositoryException if an error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getProjectMembers(String projectName, String lang) throws IllegalAccessException, AmetysRepositoryException
    {
        return getProjectMembers(projectName, lang, false);
    }
    
    /**
     * Get the members of a project, sorted by managers, non empty role and name
     * @param projectName the project's name
     * @param lang the language to get user content
     * @param expandGroup true if groups are expanded
     * @return the members of project
     * @throws AmetysRepositoryException if an error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getProjectMembers(String projectName, String lang, boolean expandGroup) throws AmetysRepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Project project = _projectManager.getProject(projectName);
        if (!_projectRightHelper.hasReadAccess(project))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }
        
        List<Map<String, Object>> membersData = new ArrayList<>();
        
        Set<ProjectMember> projectMembers = getProjectMembers(project, expandGroup);
        
        for (ProjectMember projectMember : projectMembers)
        {
            membersData.add(_member2Json(projectMember, lang));
        }
        
        result.put("members", membersData);
        result.put("success", true);
        
        return result;
    }
    
    private Map<String, Object> _member2Json(ProjectMember projectMember, String lang)
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        Map<String, Object> memberData = new HashMap<>();
        
        memberData.put("type", projectMember.getType().name().toLowerCase());
        memberData.put("title", projectMember.getTitle());
        memberData.put("sortabletitle", projectMember.getSortableTitle());
        memberData.put("manager", projectMember.isManager());
        
        String role = projectMember.getRole();
        if (StringUtils.isNotEmpty(role))
        {
            memberData.put("role", role);
        }
        
        User user = projectMember.getUser();
        if (user != null)
        {
            memberData.put("id", UserIdentity.userIdentityToString(user.getIdentity()));
            memberData.putAll(_userHelper.user2json(user));

            Content userContent = getUserContent(lang, user);
            
            if (userContent != null)
            {
                if (userContent.hasValue("function"))
                {
                    memberData.put("function", userContent.getValue("function"));
                }

                if (userContent.hasValue("organisation-accronym"))
                {
                    memberData.put("organisationAcronym", userContent.getValue("organisation-accronym"));
                }
                String usersDirectorySiteName = _projectManager.getUsersDirectorySiteName();
                String[] contentTypes = userContent.getTypes();
                for (String contentType : contentTypes)
                {
                    // Try to see if a user page exists for this content type
                    UserPage userPage = _userDirectoryPageResolver.getUserPage(userContent, usersDirectorySiteName, lang, contentType);
                    if (userPage != null)
                    {
                        memberData.put("link", _uriResolver.getResolverForType("page").resolve(userPage.getId(), false, true, false));
                    }
                }
                    
            }
            else if (getLogger().isDebugEnabled())
            {
                getLogger().debug("User content not found for user : " + user);
            }

            memberData.put("hasReadAccessOnTaskModule", _projectRightsHelper.hasReadAccessOnModule(project, TasksWorkspaceModule.TASK_MODULE_ID, user.getIdentity()));
            memberData.put("hasReadAccessOnDocumentModule", _projectRightsHelper.hasReadAccessOnModule(project, DocumentWorkspaceModule.DOCUMENT_MODULE_ID, user.getIdentity()));
            memberData.put("hasReadAccessOnForumModule", _projectRightsHelper.hasReadAccessOnModule(project, ForumWorkspaceModule.FORUM_MODULE_ID, user.getIdentity()));
            
        }
        
        Group group = projectMember.getGroup();
        if (group != null)
        {
            memberData.putAll(group2Json(group));
        }
        
        return memberData;
    }

    /**
     * Get user content
     * @param lang the lang
     * @param user the user
     * @return the user content or null if no exist
     */
    public Content getUserContent(String lang, User user)
    {
        Content userContent = _userDirectoryHelper.getUserContent(user.getIdentity(), lang);

        if (userContent == null)
        {
            userContent = _userDirectoryHelper.getUserContent(user.getIdentity(), "en");
        }
        
        if (userContent == null)
        {
            Map<String, Language> availableLanguages = _languagesManager.getAvailableLanguages();
            for (Language availableLanguage : availableLanguages.values())
            {
                if (userContent == null)
                {
                    userContent = _userDirectoryHelper.getUserContent(user.getIdentity(), availableLanguage.getCode());
                }
            }
        }
        return userContent;
    }

    /**
     * Get the members of a project, sorted by managers, non empty role and name
     * @param project the project
     * @param expandGroup true to expand the user of a group
     * @return the members of project
     * @throws AmetysRepositoryException if an error occurred
     */
    public Set<ProjectMember> getProjectMembers(Project project, boolean expandGroup) throws AmetysRepositoryException
    {
        return getProjectMembers(project, expandGroup, Set.of());
    }

    /**
     * Get the members of a project, sorted by managers, non empty role and name
     * @param project the project
     * @param expandGroup true to expand the user of a group
     * @param defaultSet default set to return when project has no site
     * @return the members of project
     * @throws AmetysRepositoryException if an error occurred
     */
    public Set<ProjectMember> getProjectMembers(Project project, boolean expandGroup, Set<ProjectMember> defaultSet) throws AmetysRepositoryException
    {
        Cache<ProjectMemberCacheKey, Set<ProjectMember>> cache = _getCache();
        if (project == null)
        {
            return defaultSet;
        }
        ProjectMemberCacheKey cacheKey = ProjectMemberCacheKey.of(project.getId(), expandGroup);
        if (cache.hasKey(cacheKey))
        {
            Set<ProjectMember> projectMembers = cache.get(cacheKey);
            return projectMembers != null ? projectMembers : defaultSet;
        }
        else
        {
            Set<ProjectMember> projectMembers = _getProjectMembers(project, expandGroup);
            cache.put(cacheKey, projectMembers);
            return projectMembers != null ? projectMembers : defaultSet;
        }
    }

    private Set<ProjectMember> _getProjectMembers(Project project, boolean expandGroup)
    {
        Comparator<ProjectMember> managerComparator = Comparator.comparing(m -> m.isManager() ? 0 : 1);
        Comparator<ProjectMember> roleComparator = Comparator.comparing(m -> StringUtils.isNotBlank(m.getRole()) ? 0 : 1);
        // Use sortable title for sort, and concatenate it with hash code of user, so that homonyms do not appear equals
        Comparator<ProjectMember> nameComparator = (m1, m2) -> (m1.getSortableTitle() + m1.hashCode()).compareToIgnoreCase(m2.getSortableTitle() + m2.hashCode());
        
        Set<ProjectMember> members = new TreeSet<>(managerComparator.thenComparing(roleComparator).thenComparing(nameComparator));
        
        Map<JCRProjectMember, Object> jcrMembers = getJCRProjectMembers(project);
        List<UserIdentity> managers = Arrays.asList(project.getManagers());
        
        Site site = project.getSite();
        if (site == null)
        {
            getLogger().error("Can not compute members in the project " + project.getName() + " because it can not be linked to an existing site");
            return null;
        }
        String projectSiteName = site.getName();

        Set<String> projectGroupDirectory = _groupDirectoryContextHelper.getGroupDirectoriesOnContext("/sites/" + projectSiteName);
        
        for (Entry<JCRProjectMember, Object> entry : jcrMembers.entrySet())
        {
            JCRProjectMember jcrMember = entry.getKey();
            if (MemberType.USER == jcrMember.getType())
            {
                User user = (User) entry.getValue();
                boolean isManager = managers.contains(jcrMember.getUser());
                
                ProjectMember projectMember = new ProjectMember(user, jcrMember.getRole(), isManager);
                if (!members.add(projectMember) && _projectManager.isUserInProjectPopulations(project, user.getIdentity()))
                {
                    //if set already contains the user, override it (users always take over users' group)
                    members.remove(projectMember); // remove the one in  the set
                    members.add(projectMember); // add the new one
                }
            }
            else if (MemberType.GROUP == jcrMember.getType())
            {
                Group group = (Group) entry.getValue();
                if (projectGroupDirectory.contains(group.getGroupDirectory().getId()))
                {
                    if (expandGroup)
                    {
                        for (UserIdentity userIdentity : group.getUsers())
                        {
                            User user = _userManager.getUser(userIdentity);
                            if (user != null && _projectManager.isUserInProjectPopulations(project, userIdentity))
                            {
                                ProjectMember projectMember = new ProjectMember(user, null, false);
                                members.add(projectMember); // add if does not exist yet
                            }
                        }
                    }
                    else
                    {
                        // Add the member as group
                        members.add(new ProjectMember(group));
                    }
                }
            }
        }
        return members;
    }
    
    /**
     * Retrieves the rights for the current user in the project
     * @param projectName The project Name
     * @return The project
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getMemberModuleRights(String projectName)
    {

        if (!_projectRightsHelper.hasReadAccessOnModule(MembersWorkspaceModule.MEMBERS_MODULE_ID))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }
        
        Map<String, Object> results = new HashMap<>();
        Map<String, Object> rights = new HashMap<>();
        
        Project project = _projectManager.getProject(projectName);
        if (project == null)
        {
            results.put("message", "unknown-project");
            results.put("success", false);
        }
        else
        {
            rights.put("view", _projectRightHelper.canViewMembers(project));
            rights.put("add", _projectRightHelper.canAddMember(project));
            rights.put("edit", _projectRightHelper.canEditMember(project));
            rights.put("delete", _projectRightHelper.canRemoveMember(project));
            results.put("rights", rights);
            results.put("success", true);
        }
        
        return results;
    }
    
    /**
     * Get the list of users of the project
     * @param project The project
     * @return The list of users
     */
    public Map<JCRProjectMember, Object> getJCRProjectMembers(Project project)
    {
        Map<JCRProjectMember, Object> projectMembers = new HashMap<>();
        
        if (project != null)
        {
            ModifiableTraversableAmetysObject membersNode = _getProjectMembersNode(project);
            
            for (AmetysObject memberNode : membersNode.getChildren())
            {
                if (memberNode instanceof JCRProjectMember)
                {
                    JCRProjectMember jCRProjectMember = (JCRProjectMember) memberNode;
                    if (jCRProjectMember.getType() == MemberType.USER)
                    {
                        UserIdentity userIdentity = jCRProjectMember.getUser();
                        User user = _userManager.getUser(userIdentity);
                        if (user != null)
                        {
                            projectMembers.put((JCRProjectMember) memberNode, user);
                        }
                    }
                    else
                    {
                        GroupIdentity groupIdentity = jCRProjectMember.getGroup();
                        Group group = _groupManager.getGroup(groupIdentity);
                        if (group != null)
                        {
                            projectMembers.put((JCRProjectMember) memberNode, group);
                        }
                    }
                }
            }
        }
        
        return projectMembers;
    }
    
    /**
     * Test if an user is a member of a project (directly or by a group)
     * @param project The project
     * @param userIdentity The user identity
     * @return True if this user is a member of this project
     */
    public boolean isProjectMember(Project project, UserIdentity userIdentity)
    {
        return getProjectMember(project, userIdentity) != null;
    }
    
    /**
     * Retrieve the member of a project corresponding to the user identity
     * @param project The project
     * @param userIdentity The user identity
     * @return The member of this project, which can be of type "user" or "group", or null if the user is not in the project
     */
    public ProjectMember getProjectMember(Project project, UserIdentity userIdentity)
    {
        return getProjectMember(project, userIdentity, null);
    }
    
    /**
     * Retrieve the member of a project corresponding to the user identity
     * @param project The project
     * @param userIdentity The user identity
     * @param userGroups The user groups. If null the user's groups will be expanded.
     * @return The member of this project, which can be of type "user" or "group", or null if the user is not in the project
     */
    public ProjectMember getProjectMember(Project project, UserIdentity userIdentity, Set<GroupIdentity> userGroups)
    {
        if (userIdentity == null)
        {
            return null;
        }
                
        Set<ProjectMember> members = getProjectMembers(project, true);
        
        ProjectMember projectMember = members.stream()
                .filter(member -> MemberType.USER == member.getType())
                .filter(member -> userIdentity.equals(member.getUser().getIdentity()))
                .findFirst()
                .orElse(null);
        
        if (projectMember != null)
        {
            return projectMember;
        }
        
        Set<GroupIdentity> groups = userGroups == null ? _groupManager.getUserGroups(userIdentity) : userGroups; // get user's groups

        if (!groups.isEmpty())
        {
            return members.stream()
                    .filter(member -> MemberType.GROUP == member.getType())
                    .filter(member -> groups.contains(member.getGroup().getIdentity()))
                    .findFirst()
                    .orElse(null);
        }
        
        return null;
    }
    
    /**
     * Set the manager of a project
     * @param projectName The project name
     * @param profileId The profile id to affect
     * @param managers The managers' user identity
     */
    public void setProjectManager(String projectName, String profileId, List<UserIdentity> managers)
    {
        Project project = _projectManager.getProject(projectName);
        if (project == null)
        {
            return;
        }
        
        project.setManagers(managers.toArray(new UserIdentity[managers.size()]));
        
        for (UserIdentity userIdentity : managers)
        {
            JCRProjectMember member = _getOrCreateJCRProjectMember(project, userIdentity);
            
            Set<String> allowedProfiles = Set.of(profileId);
            for (WorkspaceModule module : _projectManager.getModules(project))
            {
                setProfileOnModule(member, project, module, allowedProfiles);
            }
        }
        
        project.saveChanges();
        
        _getCache().invalidate(ProjectMemberCacheKey.of(project.getId(), null));
        
        // Clear rights manager cache (if I remove my own rights)
        _rightManager.clearCaches();
        
//        Request request = ContextHelper.getRequest(_context);
//        if (request != null)
//        {
//            request.removeAttribute(RightManager.CACHE_REQUEST_ATTRIBUTE_NAME);
//        }
    }
    
    private void _notifyAclUpdated(UserIdentity userIdentity, AmetysObject aclContext, Collection<String> aclProfiles)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, aclContext);
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT_IDENTIFIER, aclContext.getId());
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, aclProfiles);
        eventParams.put(org.ametys.cms.ObservationConstants.ARGS_ACL_SOLR_CACHE_UNINFLUENTIAL, true);
        
        _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, userIdentity, eventParams));
    }

    /**
     * Retrieve or create a user in a project
     * @param project The project
     * @param userIdentity the user
     * @return The user
     */
    private JCRProjectMember _getOrCreateJCRProjectMember(Project project, UserIdentity userIdentity)
    {
        Predicate<? super AmetysObject> findMemberPredicate = memberNode -> MemberType.USER == ((JCRProjectMember) memberNode).getType()
                && userIdentity.equals(((JCRProjectMember) memberNode).getUser());
        JCRProjectMember projectMember = _getOrCreateJCRProjectMember(project, findMemberPredicate);
        
        if (projectMember.needsSave())
        {
            projectMember.setUser(userIdentity);
            projectMember.setType(MemberType.USER);
        }
        
        return projectMember;
    }

    /**
     * Retrieve or create a group as a member in a project
     * @param project The project
     * @param groupIdentity the group
     * @return The user
     */
    private JCRProjectMember _getOrCreateJCRProjectMember(Project project, GroupIdentity groupIdentity)
    {
        Predicate<? super AmetysObject> findMemberPredicate = memberNode -> MemberType.GROUP == ((JCRProjectMember) memberNode).getType()
                && groupIdentity.equals(((JCRProjectMember) memberNode).getGroup());
        JCRProjectMember projectMember = _getOrCreateJCRProjectMember(project, findMemberPredicate);
        
        if (projectMember.needsSave())
        {
            projectMember.setGroup(groupIdentity);
            projectMember.setType(MemberType.GROUP);
        }
        
        return projectMember;
    }
    
    /**
     * Retrieve or create a member in a project
     * @param project The project
     * @param findMemberPredicate The predicate to find the member node
     * @return The member node. A new node is created if the member node was not found
     */
    protected JCRProjectMember _getOrCreateJCRProjectMember(Project project, Predicate<? super AmetysObject> findMemberPredicate)
    {
        ModifiableTraversableAmetysObject membersNode = _getProjectMembersNode(project);
        
        Optional<AmetysObject> member = _getProjectMembersNode(project).getChildren()
                                                                       .stream()
                                                                       .filter(memberNode -> memberNode instanceof JCRProjectMember)
                                                                       .filter(findMemberPredicate)
                                                                       .findFirst();
        if (member.isPresent())
        {
            return (JCRProjectMember) member.get();
        }
        
        String baseName = "member";
        String name = baseName;
        int index = 1;
        while (membersNode.hasChild(name))
        {
            index++;
            name = baseName + "-" + index;
        }
        
        JCRProjectMember jcrProjectMember = membersNode.createChild(name, __PROJECT_MEMBER_NODE_TYPE);
        
        // we invalidate the cache has we had to create a new user
        _getCache().invalidate(ProjectMemberCacheKey.of(project.getId(), null));
        
        return jcrProjectMember;
    }

    
    /**
     * Remove a user from a project
     * @param projectName The project name
     * @param identity The identity of the user or group, who must be a member of the project
     * @param type The type of the member, user or group
     * @return The error code, if an error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> removeMember(String projectName, String identity, String type)
    {
        Project project = _projectManager.getProject(projectName);
        if (!_projectRightHelper.canRemoveMember(project))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to remove member without convenient right [" + projectName + ", " + identity + "]");
        }
        return _removeMember(projectName, identity, type, true);
    }
    
    private Map<String, Object> _removeMember(String projectName, String identity, String type, boolean checkCurrentUser)
    {
        Map<String, Object> result = new HashMap<>();
        
        MemberType memberType = MemberType.valueOf(type.toUpperCase());
        boolean isTypeUser = MemberType.USER == memberType;
        boolean isTypeGroup = MemberType.GROUP == memberType;
        UserIdentity user = Optional.ofNullable(identity)
                                    .filter(id -> id != null && isTypeUser)
                                    .map(UserIdentity::stringToUserIdentity)
                                    .orElse(null);
        GroupIdentity group = Optional.ofNullable(identity)
                                      .filter(id -> id != null && isTypeGroup)
                                      .map(GroupIdentity::stringToGroupIdentity)
                                      .orElse(null);
        
        if (isTypeGroup && group == null
            || isTypeUser && user == null)
        {
            result.put("success", false);
            result.put("message", isTypeGroup ? "unknown-group" : "unknown-user");
            return result;
        }
        
        Project project = _projectManager.getProject(projectName);
        if (project == null)
        {
            result.put("success", false);
            result.put("message", "unknown-project");
            return result;
        }
        
        if (checkCurrentUser && _isCurrentUser(isTypeUser, user))
        {
            result.put("success", false);
            result.put("message", "current-user");
            return result;
        }
        
        // If there is only one manager, do not remove him from the project's members
        if (isTypeUser && isOnlyManager(project, user))
        {
            result.put("success", false);
            result.put("message", "only-manager");
            return result;
        }
        
        JCRProjectMember projectMember = null;
        if (isTypeUser)
        {
            projectMember = _getProjectMember(project, user);
        }
        else if (isTypeGroup)
        {
            projectMember = _getProjectMember(project, group);
        }
        
        if (projectMember == null)
        {
            result.put("success", false);
            result.put("message", "unknown-member");
            return result;
        }
        
        _removeMember(projectMember, project);
        
        result.put("success", true);
        return result;
    }

    private void _removeMember(JCRProjectMember projectMember, Project project)
    {
        _removeManager(project, projectMember);
        _removeMemberProfiles(project, projectMember);
        MemberType memberType = projectMember.getType();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_MEMBER_IDENTITY, MemberType.USER.equals(memberType)
                ? UserIdentity.userIdentityToString(projectMember.getUser())
                        : GroupIdentity.groupIdentityToString(projectMember.getGroup()));
        eventParams.put(ObservationConstants.ARGS_MEMBER_IDENTITY_TYPE, memberType);
        eventParams.put(ObservationConstants.ARGS_PROJECT, project);
        eventParams.put(ObservationConstants.ARGS_PROJECT_ID, project.getId());

        projectMember.remove();
        project.saveChanges();
        
        _getCache().invalidate(ProjectMemberCacheKey.of(project.getId(), null));
        _observationManager.notify(new Event(ObservationConstants.EVENT_MEMBER_DELETED, _currentUserProvider.getUser(), eventParams));
    }

    private boolean _isCurrentUser(boolean isTypeUser, UserIdentity user)
    {
        return isTypeUser && _currentUserProvider.getUser().equals(user);
    }

    /**
     * Check if a user is the only manager of a project
     * @param project the project
     * @param user the user
     * @return true if the user is the only manager of the project
     */
    public boolean isOnlyManager(Project project, UserIdentity user)
    {
        UserIdentity[] managers = project.getManagers();
        return managers.length == 1 && managers[0].equals(user);
    }

    private JCRProjectMember _getProjectMember(Project project, GroupIdentity group)
    {
        JCRProjectMember projectMember = null;
        ModifiableTraversableAmetysObject membersNode = _getProjectMembersNode(project);
        
        for (AmetysObject memberNode : membersNode.getChildren())
        {
            if (memberNode instanceof JCRProjectMember)
            {
                JCRProjectMember member = (JCRProjectMember) memberNode;
                if (MemberType.GROUP == member.getType() && group.equals(member.getGroup()))
                {
                    projectMember = (JCRProjectMember) memberNode;
                }

            }
        }
        return projectMember;
    }

    private JCRProjectMember _getProjectMember(Project project, UserIdentity user)
    {
        JCRProjectMember projectMember = null;
        ModifiableTraversableAmetysObject membersNode = _getProjectMembersNode(project);
        
        for (AmetysObject memberNode : membersNode.getChildren())
        {
            if (memberNode instanceof JCRProjectMember)
            {
                JCRProjectMember member = (JCRProjectMember) memberNode;
                if (MemberType.USER == member.getType() && user.equals(member.getUser()))
                {
                    projectMember = (JCRProjectMember) memberNode;
                }
            }
        }
        return projectMember;
    }
    
    private void _removeManager(Project project, JCRProjectMember projectMember)
    {
        if (MemberType.USER.equals(projectMember.getType()))
        {
            UserIdentity user = projectMember.getUser();
            UserIdentity[] oldManagers = project.getManagers();
            
            // Remove the user from the project's managers
            UserIdentity[] managers = Arrays.stream(oldManagers)
                  .filter(manager -> !manager.equals(user))
                  .toArray(UserIdentity[]::new);
            
            project.setManagers(managers);
        }
    }

    private void _removeMemberProfiles(Project project, JCRProjectMember projectMember)
    {
        for (WorkspaceModule module : _projectManager.getModules(project))
        {
            ModifiableResourceCollection moduleRootNode = module.getModuleRoot(project, false);
            _removeMemberProfiles(projectMember, moduleRootNode);
        }
    }
    
    /**
     * Retrieves the users node of the project
     * The users node will be created if necessary
     * @param project The project
     * @return The users node of the project
     */
    protected ModifiableTraversableAmetysObject _getProjectMembersNode(Project project)
    {
        if (project == null)
        {
            throw new AmetysRepositoryException("Error getting the project users node, project is null");
        }
        
        try
        {
            ModifiableTraversableAmetysObject membersNode;
            if (project.hasChild(__PROJECT_MEMBERS_NODE))
            {
                membersNode = project.getChild(__PROJECT_MEMBERS_NODE);
            }
            else
            {
                membersNode = project.createChild(__PROJECT_MEMBERS_NODE, __PROJECT_MEMBERS_NODE_TYPE);
            }
            
            return membersNode;
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the project users node", e);
        }
    }
    
    /**
     * Get the JSON representation of a group
     * @param group The group
     * @return The group
     */
    protected Map<String, Object> group2Json(Group group)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("id", GroupIdentity.groupIdentityToString(group.getIdentity()));
        infos.put("groupId", group.getIdentity().getId());
        infos.put("label", group.getLabel());
        infos.put("sortablename", group.getLabel());
        infos.put("groupDirectory", group.getIdentity().getDirectoryId());
        return infos;
    }

    /**
     * Count the total of unique users in the project and in the project's group
     * @param project The project
     * @return The total of members
     */
    public Long getMembersCount(Project project)
    {
        Set<ProjectMember> projectMembers = getProjectMembers(project, true);
                
        return (long) projectMembers.size();
    }

    /**
     * Get the users from a group that are part of the project. They can be filtered with a predicate
     * @param group The group
     * @param project The project
     * @param filteringPredicate The predicate to filter
     * @return The list of users
     */
    public List<User> getGroupUsersFromProject(Group group, Project project, BiPredicate<Project, UserIdentity> filteringPredicate)
    {
        Set<String> projectPopulations = _populationContextHelper.getUserPopulationsOnContexts(List.of("/sites/" + project.getSite().getName()), false, false);
        
        return group.getUsers().stream()
                .filter(user -> projectPopulations.contains(user.getPopulationId()))
                .filter(user -> filteringPredicate.test(project, user))
                .map(_userManager::getUser)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * Make the current user leave the project
     * @param projectName The project name
     * @return The error code, if an error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> leaveProject(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        if (!isProjectMember(project, _currentUserProvider.getUser()))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried perform operation without convenient right");
        }
        UserIdentity currentUser = _currentUserProvider.getUser();
        String identity = UserIdentity.userIdentityToString(currentUser);
        
        Map<String, Object> result = _removeMember(projectName, identity, MemberType.USER.toString(), false);
        
        return result;
    }
    
    private Cache<ProjectMemberCacheKey, Set<ProjectMember>> _getCache()
    {
        return _abstractCacheManager.get(__PROJECT_MEMBER_CACHE);
    }
    
    /**
     * This class represents a member of a project. Could be a user or a group
     *
     */
    public static class ProjectMember
    {
        private String _title;
        private String _sortableTitle;
        private MemberType _type;
        private String _role;
        private User _user;
        private Group _group;
        private boolean _isManager;
        
        /**
         * Create a project member as a group
         * @param group the group attached to this member. Cannot be null.
         */
        public ProjectMember(Group group)
        {
            _title = group.getLabel();
            _sortableTitle = group.getLabel();
            _type = MemberType.GROUP;
            _role = null;
            _isManager = false;
            _user = null;
            _group = group;
        }
        
        /**
         * Create a project member as a group
         * @param role the role
         * @param isManager true if the member is a manager of the project
         * @param user the user attached to this member. Cannot be null.
         */
        public ProjectMember(User user, String role, boolean isManager)
        {
            _title = user.getFullName();
            _sortableTitle = user.getSortableName();
            _type = MemberType.USER;
            _role = role;
            _isManager = isManager;
            _user = user;
            _group = null;
        }
        
        /**
         * Get the title of the member.
         * @return The title of the member
         */
        public String getTitle()
        {
            return _title;
        }
        
        /**
         * Get the sortable title of the member.
         * @return The sortable title of the member
         */
        public String getSortableTitle()
        {
            return _sortableTitle;
        }

        /**
         * Get the type of the member. It can be a user or a group
         * @return The type of the member
         */
        public MemberType getType()
        {
            return _type;
        }

        /**
         * Get the role of the member.
         * @return The role of the member
         */
        public String getRole()
        {
            return _role;
        }

        /**
         * Test if the member is a manager of the project
         * @return True if this user is a manager of the project
         */
        public boolean isManager()
        {
            return _isManager;
        }
        
        /**
         * Get the user of the member.
         * @return The user of the member
         */
        public User getUser()
        {
            return _user;
        }
        
        /**
         * Get the group of the member.
         * @return The group of the member
         */
        public Group getGroup()
        {
            return _group;
        }
        
        @Override
        public boolean equals(Object obj)
        {
            if (obj == null || !(obj instanceof ProjectMember))
            {
                return false;
            }
            
            ProjectMember otherMember = (ProjectMember) obj;
            
            if (getType() != otherMember.getType())
            {
                return false;
            }
            
            if (getType() == MemberType.USER)
            {
                return getUser().equals(otherMember.getUser());
            }
            else
            {
                return getGroup().equals(otherMember.getGroup());
            }
        }
        
        @Override
        public int hashCode()
        {
            return getType() == MemberType.USER ? getUser().getIdentity().hashCode() : getGroup().getIdentity().hashCode();
        }
    }
    
    private static final class ProjectMemberCacheKey extends AbstractCacheKey
    {
        public ProjectMemberCacheKey(String projectId, Boolean extendGroup)
        {
            super(projectId, extendGroup);
        }
        
        public static ProjectMemberCacheKey of(String projectId, Boolean withExpandedGroup)
        {
            return new ProjectMemberCacheKey(projectId, withExpandedGroup);
        }
    }
    
    public int getPriority()
    {
        return Integer.MAX_VALUE;
    }
    
    public boolean supports(Event event)
    {
        String eventId = event.getId();
        return org.ametys.core.ObservationConstants.EVENT_USER_DELETED.equals(eventId)
            || org.ametys.core.ObservationConstants.EVENT_GROUP_DELETED.equals(eventId);
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        // handle manually the removal of member nodes.
        // All method providing access to project member needs to resolve the User or Group
        String query;
        switch (event.getId())
        {
            case org.ametys.core.ObservationConstants.EVENT_USER_DELETED:
                UserIdentity userIdentity = (UserIdentity) event.getArguments().get(org.ametys.core.ObservationConstants.ARGS_USER);
                query = "//element(*, " + JCRProjectMemberFactory.MEMBER_NODETYPE + ")["
                        + new UserExpression(JCRProjectMember.METADATA_USER, Operator.EQ, userIdentity).build() + "]";
                break;
            case org.ametys.core.ObservationConstants.EVENT_GROUP_DELETED:
                GroupIdentity groupIdentity = (GroupIdentity) event.getArguments().get(org.ametys.core.ObservationConstants.ARGS_GROUP);
                query = "//element(*, " + JCRProjectMemberFactory.MEMBER_NODETYPE + ")["
                    + JCRProjectMember.METADATA_GROUP + "/" + JCRProjectMember.METADATA_GROUP_ID + "=" + groupIdentity.getId()
                    + " and " + JCRProjectMember.METADATA_GROUP + "/" + JCRProjectMember.METADATA_GROUP_DIRECTORY + "=" + groupIdentity.getDirectoryId()
                    + "]";
                break;
            default:
                throw new IllegalStateException("Event id '" + event.getId() + "' is not supported");
        }
        
        try (AmetysObjectIterable<JCRProjectMember> members = _resolver.query(query))
        {
            for (JCRProjectMember member: members)
            {
                AmetysObject parent = member.getParent().getParent();
                if (parent instanceof Project project)
                {
                    _removeMember(member, project);
                }
            }
        }
    }
}
