/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.helper;

import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.content.indexing.solr.SolrResourceGroupedMimeTypes;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.workspaces.WorkspacesHelper.FileType;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarColorsComponent;
import org.ametys.plugins.workspaces.calendars.CalendarColorsComponent.CalendarColor;
import org.ametys.plugins.workspaces.tasks.Task;

/**
 * Helper providing information required for mail notification
 */
public class MailXSLTHelper implements Serviceable
{
    private static CalendarColorsComponent _calendarColors;
    private static AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _calendarColors = (CalendarColorsComponent) manager.lookup(CalendarColorsComponent.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    
    // TODO there should be a better handling of what happens when the data is not available in all those method
    /**
     * Get a file type from a resourceId.
     * @param resourceId the id of the resource we want to get the file type
     * @return the name of the file type
     */
    public static String getFileTypeFromId(String resourceId)
    {
        try
        {
            JCRResource resource = _resolver.resolveById(resourceId);
            return getFileType(resource.getMimeType());
        }
        catch (Exception e)
        {
            return FileType.UNKNOWN.name().toLowerCase();
        }
    }

    
    /**
     * Get a file type from a mime type.
     * @param mimeType the mime type to transform
     * @return the name of the file type
     */
    public static String getFileType(String mimeType)
    {
        Optional<String> group = SolrResourceGroupedMimeTypes.getGroup(mimeType);
        
        return group.map(groupMimeType -> groupMimeType.toLowerCase())
                    .orElse(FileType.UNKNOWN.name().toLowerCase());
    }
    
    /**
     * Get the color of a {@link Calendar} from its id
     * @param calendarId the calendar's id
     * @return the color of the calendar
     */
    public static String getCalendarColor(String calendarId)
    {
        try
        {
            Calendar calendar = _resolver.resolveById(calendarId);
            CalendarColor color = _calendarColors.getColor(calendar.getColor());
            return color.getColor();
        }
        catch (UnknownAmetysObjectException e)
        {
            return _calendarColors.getColors().values().iterator().next().getColor();
        }
    }
    
    /**
     * Get the ics export link of a calendar event from its id
     * @param eventId the event's id
     * @param projectURL the URL of the project
     * @return the recurrence of the calendar event
     */
    public static String getCalendarEventICSExportLink(String eventId, String projectURL)
    {
        // Try to resolve the event id to check if the event is deleted or not
        if (_resolver.hasAmetysObjectForId(eventId))
        {
            return projectURL + "/_plugins/workspaces/event.ics?eventId=" + eventId;
        }
        else
        {
            return "";
        }
    }
    
    /**
     * Test if a task have a due date from its id
     * @param taskId the task's id
     * @return true if the task has a due date
     */
    public static boolean hasDueDate(String taskId)
    {
        try
        {
            Task task = _resolver.resolveById(taskId);
            return task.getDueDate() != null;
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
    }

    /**
     * Get the ics export link of a task from its id
     * @param taskId the task's id
     * @param projectURL the URL of the project
     * @return the recurrence of the calendar event
     */
    public static String getTaskICSExportLink(String taskId, String projectURL)
    {
        return projectURL + "/_plugins/workspaces/task.ics?id=" + taskId;
    }
}
