/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.observers;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.site.Site;

/**
 * Abstract observer to have code executed when a sitemap is updated in a project
 */
public abstract class AbstractInitializeProjectObserver extends AbstractLogEnabled implements Observer, Serviceable, PluginAware
{

    /** Project Manager */
    protected ProjectManager _projectManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        // Listen on site updated because it is too early to listen on sitemap added (we need the site be configured and synchronized in live)
        return event.getId().equals(ObservationConstants.EVENT_SITE_UPDATED);
    }

    @Override
    public int getPriority()
    {
        // Will be processed after live synchronization observers
        return MAX_PRIORITY + 2000;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Site site = (Site) event.getArguments().get(ObservationConstants.ARGS_SITE);
        if (site != null)
        {
            List<Project> projects = _projectManager.getProjectsForSite(site);
            if (!projects.isEmpty())
            {
                Project project = projects.get(0);
                
                if (project != null)
                {
                    doObserve(event, transientVars, site, project);
                }
            }
        }
    }
    
    /**
     * This method will be called by {@link Observer#observe(Event, Map)} when there is a project on this site
     * @param event the event
     * @param transientVars transientVars passed from one Observer to another when processing a single Event. 
     * This may allow optimizations between observers.
     * @param site the site impacted
     * @param project the project impacted
     * @throws Exception something went wrong
     */
    protected abstract void doObserve(Event event, Map<String, Object> transientVars, Site site, Project project) throws Exception;
}
