/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.report;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Component for reporting projects
 */
public class ReportHelper implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = ReportHelper.class.getName();
    
    /** The right id to report all project */
    public static final String REPORT_ALL_RIGHT_ID = "Workspaces_Reports_Right_Report_All";
    
    /** The id of reports service */
    public static final String REPORT_SERVICE_ID = "org.ametys.plugins.workspacesReports.service.Report";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The project manager */
    protected ProjectManager _projectManager;
    
    /** The right manager */
    protected RightManager _rightManager;

    /** The catgeory provider */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
    }
    
    /**
     * Get the available projects (the manager's project)
     * @return the available projects for reports
     */
    public List<Project> getAvailableProjects()
    {
        UserIdentity user = _currentUserProvider.getUser();
        boolean hasSuperRight = hasSuperRight();

        Function<Project, String> getProjectTitle = Project::getTitle;
        Comparator<Project> projectTitleComparator = Comparator.comparing(getProjectTitle.andThen(StringUtils::stripAccents), String.CASE_INSENSITIVE_ORDER);
        
        return _projectManager.getProjects()
            .stream()
            .filter(p -> hasSuperRight || ArrayUtils.contains(p.getManagers(), user))
            .sorted(projectTitleComparator)
            .collect(Collectors.toList());
    }
    
    /**
     * Determines if the current user has right to access to all project's reports
     * @return if the current user has right to access to all project's reports
     */
    public boolean hasSuperRight()
    {
        return hasSuperRight(_currentUserProvider.getUser());
    }
    
    /**
     * Determines if a user has right to access to all project's reports
     * @param user the user to check
     * @return if the user has right to access to all project's reports
     */
    public boolean hasSuperRight(UserIdentity user)
    {
        return _rightManager.hasRight(user, REPORT_ALL_RIGHT_ID, "/cms") == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Get the available projects (the manager's project)
     * @param filteredProjects the filtered projects
     * @param filteredCategories the filtered categories
     * @return the available projects for reports
     */
    public List<Project> getAvailableProjects(List<String> filteredProjects, List<String> filteredCategories)
    {
        return getAvailableProjects()
            .stream()
            .filter(p -> filteredProjects == null || filteredProjects.isEmpty() || filteredProjects.contains(p.getId()))
            .filter(p -> _matchCategories(p, filteredCategories))
            .collect(Collectors.toList());
    }
    
    private boolean _matchCategories(Project project, List<String> filteredCategories)
    {
        if (filteredCategories == null || filteredCategories.isEmpty())
        {
            return true;
        }
        
        for (String projectCategory : project.getCategories())
        {
            Category category = _categoryProviderEP.getTag(projectCategory, null);
            if (category != null)
            {
                Category rootCategory = _getRootCategory(category);
                if (filteredCategories.contains(rootCategory.getName()))
                {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get the available root categories
     * @param projects the projects
     * @return the available categories from the given projects
     */
    public Set<Category> getAvailableCategories(List<Project> projects)
    {
        return projects.stream()
            .map(Project::getCategories)
            .flatMap(Collection::stream)
            .map(id -> _categoryProviderEP.getTag(id, null))
            .filter(Objects::nonNull)
            .map(c -> _getRootCategory(c))
            .collect(Collectors.toSet());
    }
    
    private Category _getRootCategory(Category category)
    {
        Category parent = category;
        while (parent.getParent() != null)
        {
            parent = parent.getParent();
        }
        return parent;
    }
}
