/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle actions on project tags.
 * @private
 */
Ext.define('Ametys.plugins.workspaces.project.category.CategoryActions', {
    singleton : true,

    /**
     * Function called to add a new tag
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    add: function(controller)
    {
    	var targets = controller.getMatchingTargets();
    	if (targets.length > 0)
        {
    		var parentTagId = targets[0].getParameters().id;
    		if (!this._formPanel)
    		{
    			this._loadColor(Ext.bind(this._add, this, [parentTagId]));
    		}
    		else
    		{
    			this._add(parentTagId);
    		}
        }  
    },
    
    /**
	 * Create a tag
	 * @param {String} parentTagId The parent tag id of the created tag
	 * @param {Function} callback The callback
	 * @private
	 */
    _add: function(parentTagId, callback)
    {
    	Ametys.plugins.cms.tag.TagHelper.add(
    			parentTagId,
				this._formPanel,
				Ametys.message.MessageTarget.PROJECT_CATEGORY,
				"org.ametys.plugins.workspaces.categories.CategoryJCRDAO",
				"{{i18n PLUGINS_WORKSPACES_DIALOGBOX_CATEGORIES_ADD_LABEL}}",
				"ametysicon-folder250 decorator-ametysicon-add64",
				callback
		);
    },
    
    /**
     * Function called to edit a tag
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    edit: function (controller)
    {
    	var targets = controller.getMatchingTargets();
    	if (targets.length > 0)
        {
    		var tagId = targets[0].getParameters().id;
    		if (!this._formPanel)
    		{
    			this._loadColor(Ext.bind(this._edit, this, [tagId]));
    		}
    		else
    		{
    			this._edit(tagId);
    		}
        }  
    },
    
    /**
	 * Edit a tag
	 * @param {String} tagId The tag id to edit
	 * @param {Function} callback The callback
	 * @private
	 */
    _edit: function(tagId, callback)
    {
    	Ametys.plugins.cms.tag.TagHelper.edit(
        		tagId,
        		this._formPanel,
        		Ametys.message.MessageTarget.PROJECT_CATEGORY,
				"org.ametys.plugins.workspaces.categories.CategoryJCRDAO",
				"{{i18n PLUGINS_WORKSPACES_DIALOGBOX_CATEGORIES_EDIT_LABEL}}",
        		"ametysicon-folder250 decorator-ametysicon-edit45",
        		callback
		);
    },
    
    /**
     * Function called to remove a tag
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    remove: function (controller)
    {
    	var targets = controller.getMatchingTargets();
        if (targets.length > 0)
        {
            var tagId = targets[0].getParameters().id;
            Ametys.plugins.cms.tag.TagHelper.remove(
            		tagId, 
            		Ametys.message.MessageTarget.PROJECT_CATEGORY,
    				"org.ametys.plugins.workspaces.categories.CategoryJCRDAO",
    				"{{i18n PLUGINS_WORKSPACES_DIALOGBOX_CATEGORIES_REMOVE_LABEL}}",
            		"{{i18n PLUGINS_WORKSPACES_DIALOGBOX_CATEGORIES_REMOVE_DESC}}"
    		);
        }
    },
    
    /**
	 * Get the colors
	 * @param {Function} callback The callback function
	 * @private
	 */
    _loadColor: function (callback)
	{
		Ametys.data.ServerComm.callMethod({
			role: "org.ametys.plugins.workspaces.categories.CategoryColorsComponent",
			methodName: "getColors",
			parameters: [],
			callback: {
				scope: this,
				handler: Ext.bind(this._loadColorCb, this, [callback], 1),
			},
			errorMessage: true
		});
	},
    
    /**
	 * Load the colors types data into the store
	 * @param {Object[]} colors The list of color object
	 * @param {Function} callback The callback
	 * @private
	 */
	_loadColorCb: function (colors, callback)
	{
		var data = [];
		
		var colorList = [];
		for (var i in colors)
		{
			var color = colors[i];
			var colorObj = [];
			colorObj.push(i);
			colorObj.push(color.main);
			
			colorList.push(colorObj);
		}
		
		var colorField = Ext.create("Ametys.form.widget.ColorPicker", {
        	name: 'color',
        	fieldLabel: '{{i18n PLUGINS_WORKSPACES_FORM_CATEGORY_COLOR_LABEL}}',
        	ametysDescription: "{{i18n PLUGINS_WORKSPACES_FORM_CATEGORY_COLOR_DESC}}",
        	enumeration: colorList
        })
		
        this._initFormPanel(colorField);
		
		if (callback) callback();
	},
    
	/**
	 * Init the form panel
	 * @private
	 */
	_initFormPanel: function(colorField)
    {
		this._formPanel = Ext.create('Ext.form.Panel', {
            border: false,
            defaultType :'textfield',
            
            defaults: {
                cls: 'ametys',
                labelWidth: 90,
                labelAlign: 'right',
                labelSeparator: '',
                msgTarget: 'side',
                width: 340
            },
            
            items: [{
                        xtype: 'hidden',
                        name: 'id'
                    },
                    {
                        xtype: 'hidden',
                        name: 'parentID'
                    },
                    {
                        xtype:'textfield',
                        name: 'title',
                        itemId: 'title',
                        fieldLabel: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_TITLE}}",
                        ametysDescription: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_TITLE_DESCRIPTION}}",
                        allowBlank: false,
                        enableKeyEvents: true,
                        listeners: {
                            'keyup': Ext.bind(Ametys.plugins.cms.tag.TagHelper.autoFillKey, this)
                        }
                    },
                    {
                        xtype: 'hidden',
                        name: 'name',
                        fieldLabel: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_NAME}}",
                        allowBlank: false,
                        regex: /^[A-Za-z0-9_-]+$/,
                        regexText: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_INVALID_REGEXP}}"
                    },
                    {
                    	xtype:'textarea',
                    	name: 'description',
                    	fieldLabel: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_DESC}}",
                    	ametysDescription: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_DESC_DESCRIPTION}}",
                    	height: 70
                    },
                    colorField
                ]
		});
    }
});