/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
AmetysRouter = {
    
    /** The separator between the type and the id */
    _hashSeparator: "-",
    
    /** The map of route configuration for each type */
    _routesConfiguration: {},
    
    /**
     * For a given type, call the OnHashChangeFn in the linked configuration
     * @param {String} type the hash type
     * @param {String} id The hash id
     * @private
     */
    _callOnHashChangeFn: function(type, id)
    {
        if (AmetysRouter._routesConfiguration[type])
        {
            AmetysRouter._routesConfiguration[type].onHashChangeFn(id);
        }
    },
    
    /**
     * Call all the OnHashClearFn in the configuration
     * @param {Object} params the params to pass to hash clear functions
     * @private
     */
    _callOnHashClearFns: function(params)
    {
        for (var type in AmetysRouter._routesConfiguration)
        {
            AmetysRouter._routesConfiguration[type].onHashClearFn(params);
        }
    },
    
    /**
     * Parse the given hash
     * @param {String} hash The hash to parse
     * @return {Object} hash The hash object
     * @return {String} hash.id The object id in the hash
     * @return {String} hash.type The type of object in the hash
     * @private
     */
    _parseHash: function(hash)
    {
        var index = hash.indexOf(AmetysRouter._hashSeparator);
        var hashId = hash.substr(index + AmetysRouter._hashSeparator.length);
        var type = hash.substr(1, index - 1); 

        return {"id": hashId, "type": type}
    },
    
    /**
     * Build the hash
     * @param {String} type The hash type
     * @param {String} id The hash id
     * @return {String} The hash
     * @private
     */
    _buildHash: function(type, id)
    {
        return "#" + type + AmetysRouter._hashSeparator + id;
    },
    
    /**
     * Get the current url
     * @return {String} The current url
     * @private
     */
    _getCurrentUrl: function()
    {
        return window.location.pathname + window.location.search;
    },
    
    /**
     * Register the route for a list of given type and function to calculate the hash.
     * @param {String} type The type. Only hash which begin by the type triggers _routesConfiguration.onHashChangeFn
     * @param {Object} routeConfiguration the route configuration
     * @param {Function} routeConfiguration.onHashChangeFn The function to apply when the hash changes
     * @param {Function} routeConfiguration.onHashClearFn The function to apply when the hash becomes empty
     */
    registerRoute: function (type, routeConfiguration)
    {
        AmetysRouter._routesConfiguration[type] = routeConfiguration;
        
        $(document).ready(function() {
            var hash = window.location.hash;
            if (hash)
            {
                // First start, remove the existing hash in the url
                history.replaceState(null, document.title, AmetysRouter._getCurrentUrl());
                
                var hash = AmetysRouter._parseHash(hash);
                AmetysRouter.changeRoute(hash.type, hash.id);
            }
        });
        
        // Listening the hash change
        $(window).on('hashchange', function()
        {
            var hash = window.location.hash;
            if (AmetysRouter._backing)
            {
                history.replaceState(null, document.title, AmetysRouter._getCurrentUrl());
                AmetysRouter._backing = false;
                return;
            }
            
            AmetysRouter._backing = false;
            if (hash)
            {
                var hash = AmetysRouter._parseHash(hash);
                AmetysRouter.changeRoute(hash.type, hash.id, true);
            }
            else
            {
                AmetysRouter.clearRoute();
            }
        });
    },
    
    /**
     * Change the hash for a given type
     * @param {String} type the type
     * @param {String} id the new value for hash
     */
    changeRoute: function(type, id, changeRouteByHash) 
    {
        
        AmetysRouter.changeRouteByHash = changeRouteByHash;
        AmetysRouter._backing = false;
        
        if (history.state)
        {
            history.replaceState(type, document.title, AmetysRouter._getCurrentUrl() + AmetysRouter._buildHash(type, id));
        }
        else
        {
            history.pushState(type, document.title, AmetysRouter._getCurrentUrl() + AmetysRouter._buildHash(type, id));
        }
        
        AmetysRouter._callOnHashChangeFn(type, id)
    },
    
    /**
     * Clear the hash
     * @param {Object} params the params to pass to hash clear functions
     */
    clearRoute: function(params) 
    {
        AmetysRouter._backing = false;
        if (history.state)
        {
            // We can either come here by vue code or by anchor/url change
            // only set backing to true on the former case as otherwise the hashchange function will return to early
            if (AmetysRouter.changeRouteByHash)
            {
                AmetysRouter._backing = true;
            }
            history.back();
        }
        AmetysRouter.changeRouteByHash = false;

        history.replaceState(null, document.title, AmetysRouter._getCurrentUrl());
        
        AmetysRouter._callOnHashClearFns(params);
    }
}
