/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import { callMethod } from '@common/helper/ServerCommHelper'
import AmetysFront from 'AmetysFront'

/**
 * Pinia store for managing file activities.
 */
export const useFileActivityStore = defineStore('fileActivity', {
    state: () => (
    {
        activities: [], // The ativity of the current file
        loaded: false, // Whether the activities has been loaded
        loadBy: 4, // The number of activities to load at once
        allActivitiesCount: 0 // The total count of all activities
    }),
    actions:
    {
        /**
         * Clears the activities list.
         */
        clearActivities()
        {
            this.activities = []
        },
        
        /**
         * Sets the activities list.
         * @param {Array} activities
         */
        setActivities(activities)
        {
            this.activities = activities
        },
        
        /**
         * Sets the number of activities to load.
         * @param {number} count
         */
        setLoadBy(count)
        {
            this.loadBy = count
        },
        
        /**
         * Sets the total count of all activities.
         * @param {number} count
         */
        setAllActivitiesCount(count)
        {
            this.allActivitiesCount = count
        },
        
        /**
         * Loads activities for a given item.
         * @param {Object} item
         */
        async loadActivities(item)
        {
            this.clearActivities()
            if (item && item.type !== 'folder')
            {
                let activities = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'resourceHistory',
                    parameters: [item.id]
                })
                activities.forEach(activity =>
                {
                    activity.readableCreatedAt = AmetysFront.Utils.toRelativeDate(activity.createdAt, {withTime: true});
                })
                this.setAllActivitiesCount(activities.length)
                this.setActivities(activities.slice(0, this.loadBy))
            }
        },
        
        /**
         * Restores a version and reloads activities.
         * @param {Object} payload
         */
        async restoreVersion(payload)
        {
            await callMethod(
            {
                role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                methodName: 'restoreResource',
                parameters: [payload.item.id, payload.rawVersionName]
            })
            await this.loadActivities(payload.item)
        }
    }
})