/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront'
import { callMethod } from '@common/helper/ServerCommHelper'
import i18n from 'i18n'
import { useFilesStore } from '@/stores/files'
import { useFoldersStore } from '@/stores/folders'

/**
 * Pinia store for filtering files and folders
 */
export const useFiltersStore = defineStore('filters', {
    state: () => ({
        filterByName: '', // Filter by name
        filterByType: '', // Filter by type
        filterByTag: {}, // Filter by tag
        allSystem: false, // Used for mobile to know if it is a filter for files and folder or a global search
        hideFolders: false, // Hide folders when a filter is applied
        searchActive: false, // Indicates if a search is active
        loading: false, // Loading state for the filter
    }),
    
    getters: {
        /**
         * Check if filter is active (folders are hidden)
         * @returns {boolean}
         */
        isFilter: state =>
        {
            return state.hideFolders
        },

        /**
         * Check if filter is by tag
         * @returns {boolean}
         */
        filterIsTag: state =>
        {
            return Object.keys(state.filterByTag).length > 0
        }
    },
    actions:
    {
        /**
         * Search by type
         * @param {Object} item the item to search by type
         * @param {string} item.name the name of the type
         * @param {string} item.type the type of the item
         */
        async searchByType(item)
        {

            const filesStore = useFilesStore()
            const foldersStore = useFoldersStore()
            this.searchActive = true
            this.hideFolders = true
            this.filterByType = {
                name: item.name,
                type: item.type
            }

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'searchFilesByType',
                    parameters: [item.type]
                })

                filesStore.setFiles(result.resources)
                foldersStore.setTreeActiveFolder(null)
                AmetysFront.Event.fire('reset-filter')
            }
            catch (e)
            {
                foldersStore.setTreeActiveFolder(null)
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_SEARCH_ERROR,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: e
                })
            }
        },

        /**
         * Search files by name
         * @param {string} name the name to search for
         */
        async searchByName(name)
        {
            const filesStore = useFilesStore();
            const foldersStore = useFoldersStore();
            this.searchActive = true
            this.hideFolders = true

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'searchFiles',
                    parameters: [name, AmetysFront.getAppParameter("sitemapLanguage")]
                })

                filesStore.setFiles(result.resources)
                foldersStore.setTreeActiveFolder(null)
            }
            catch (e)
            {
                foldersStore.setTreeActiveFolder(null)
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_SEARCH_ERROR,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: e
                })
            }
        },

        /**
         * Search files by name
         * @param {string} name the name to search for
         */
        async setByTag(tag)
        {
            this.allSystem = false;
            this.hideFolders = true;
            this.filterByTag = tag;
        },

        /**
         * Search files by name
         * @param {boolean} allSystem True to set the filter to all system files, false otherwise
         */
        async setAllSystem(allSystem)
        {
            this.allSystem = allSystem;
        },
        
        /**
         * Reset all filters
         * @param {boolean} reload whether to reload the folder
         */
        resetFilters(reload)
        {
            const foldersStore = useFoldersStore();
            this.filterByName = ''
            this.filterByType = ''
            this.filterByTag = {}
            this.hideFolders = false
            this.searchActive = false

            if (reload)
            {
                foldersStore.loadRootFolder()
            }
        },
        
        /**
         * Set the loading state
         * @param {boolean} loading the loading state
         */
        setLoading(loading)
        {
            this.loading = loading
        }
    }
})
