/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';

/**
 * Pinia store for task lists
 */
export const useTasksListStore = defineStore('tasksList', {
    state: () => ({
        tasksLists: [], // the tasks lists
        tasksListColors: [], // the list of available colors for tasks lists
        tasksListIcons: [] // the list of available icons for tasks lists
    }),
    getters: 
    {
        /**
         * Get the tasks lists with colors and icons
         * @returns {Array} the tasks lists with colors and icons
         */
        getTasksLists: state => 
        {
            const colors = state.tasksListColors || [];
            const icons = state.tasksListIcons || [];
            const data = state.tasksLists.map((t) => 
            {
                t.color = colors.find(c=> c.id == t.colorId) || null;
                t.icon = icons.find(c=> c.id == t.iconId) || null;
                return t; 
            }); 
            return data;
        },
    },
    actions:
    {
        /**
         * Load the tasks lists
         */
        async loadTasksLists()
        {
            AmetysFront.Event.fire('loader', true)

            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'getTasksLists',
                parameters: []
            })
            .then(tasksLists =>{
                this.tasksLists = tasksLists;
                AmetysFront.Event.fire('loader', false)
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASKS_LIST_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                });
            })
        },

        /**
         * Add a new tasks list
         * @param {String} label the label of the tasks list
         */
        async addTasksList(label)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_ADD
            });
            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'addTasksList',
                parameters: [
                    label
                ]
            })
            .then(tasksLists => {
                this.tasksLists.push(tasksLists);
            
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_ADD
                });
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                });
            })
        },

        /**
         * Delete a tasks list
         * @param {String} id the id of the tasks list to delete
         */
        async deleteTasksList(id)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_DELETE
            });

            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'deleteTasksList',
                parameters: [id]
            })
            .then(() => {
                this.tasksLists = this.tasksLists.filter(tasksList => tasksList.id != id);
                
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_DELETE
                });
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_DELETE_TASKS_LIST_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                });
            })
        },
        
        /**
         * Edit a tasks list name
         * @param {String} id the id of the tasks list to edit
         * @param {String} newLabel the new label of the tasks list
         */
        async editTasksList(id, newLabel)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_EDIT
            });

            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'editTasksList',
                parameters: [
                    id,
                    newLabel
                ]
            })
            .then(data => 
            {
                this.tasksLists.forEach(t => {
                    if (t.id === data.id)
                    {
                        t.label = newLabel;
                    }
                });
    
                AmetysFront.Event.fire('loaderEnd', 
                {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_EDIT
                });
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_EDIT_TASKS_LIST_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                });
            })
        },
        
        /**
         * Move a tasks list to a new position
         * @param {String} id the id of the tasks list to move
         * @param {Number} newPosition the new position of the tasks list
         */
        async moveTasksList(id, newPosition) {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_MOVE
            });

            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'moveTasksList',
                parameters: [
                    id,
                    newPosition
                ]
            })
            .then(data =>
            {
                this.tasksLists.forEach(t =>
                {
                    if (t.id === data.id)
                    {
                        t.position = newPosition;
                    }
                });
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_MOVE
                });
            })
            .catch(error =>
            {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_MOVE_TASKS_LIST_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                });
            })
        },

        /**
         * Load the tasks list colors
         */
        async loadTasksListColors()
        {
            AmetysFront.Event.fire('loader', true)

            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'getTasksListColors',
                parameters: []
            }) 
            .then(tasksListColors => {
                this.tasksListColors = tasksListColors;
                AmetysFront.Event.fire('loader', false)
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASKS_LIST_COLORS_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                }); 
            })
        },

        /**
         * Update the color of a tasks list
         * @param {String} id the id of the tasks list to update
         * @param {String} colorId the id of the new color
         */
        async updateTasksListColor(id, colorId)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_COLOR_EDIT
            }); 
            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'updateTasksListColor',
                parameters: [
                    id,
                    colorId
                ]
            })
            .then(() =>
            {
                this.tasksLists.forEach(t =>
                {
                    if (t.id === id)
                    {
                        t.colorId = colorId;
                    }
                });
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_COLOR_EDIT
                });
            })
            .catch(error =>
            {
                AmetysFront.Event.fire('loaderFail',
                {
                title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_TASKS_LIST_COLOR_EDIT_ERROR_MSG,
                text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                details: error
                });
            })
        },

        /**
         * Load the tasks list icons
         */
        async loadTasksListIcons()
        {
            AmetysFront.Event.fire('loader', true)

            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'getTasksListIcons',
                parameters: []
            }) 
            .then(tasksListIcons => {
                this.tasksListIcons = tasksListIcons;
                AmetysFront.Event.fire('loader', false)
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_LOAD_TASKS_LIST_ICONS_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                }); 
            })
        },

        /**
         * Update the icon of a tasks list
         * @param {String} id the id of the tasks list to update
         * @param {String} iconId the id of the new icon
         */
        async updateTasksListIcon(id, iconId)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_BEGIN_ICON_EDIT
            });
            await callMethod({
                role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO',
                methodName: 'updateTasksListIcon',
                parameters: [
                    id,
                    iconId
                ]
            })
            .then(() =>
            {
                this.tasksLists.forEach(t =>
                {
                    if (t.id === id)
                    {
                        t.iconId = iconId;
                    }
                });
                AmetysFront.Event.fire('loaderEnd',
                {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASKS_LIST_LOADING_END_ICON_EDIT
                });
            })
            .catch(error =>
            {
                AmetysFront.Event.fire('loaderFail',
                { 
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_TASKS_LIST_ICON_EDIT_ERROR_MSG,
                    text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                    details: error
                });
            })
        },
    }
})

