/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.activitystream;

import java.io.IOException;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;

/**
 * Generator for activity stream and projects services
 *
 */
public class WorkspacesServiceGenerator extends ServiceableGenerator
{
    private ProjectManager _projectManager;
    private CategoryHelper _categoryHelper;
    private CurrentUserProvider _currentUserProvider;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(manager);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _categoryHelper = (CategoryHelper) smanager.lookup(CategoryHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "Workspaces");
        
        UserIdentity user = _currentUserProvider.getUser();
        
        boolean saxCategories = parameters.getParameterAsBoolean("saxCategories", false);
        if (saxCategories)
        {
            _saxCategories(user);
        }
        
        boolean saxModules = parameters.getParameterAsBoolean("saxModules", false);
        if (saxModules)
        {
            _saxModules(user);
        }
        
        XMLUtils.endElement(contentHandler, "Workspaces");
        
        contentHandler.endDocument();
    }

    private void _saxCategories(UserIdentity user) throws SAXException
    {
        Set<Category> categories = _projectManager.getUserProjectsCategories(user);
        XMLUtils.startElement(contentHandler, "categories");
        for (Category category : categories)
        {
            _saxCategory(category);
        }
        XMLUtils.endElement(contentHandler, "categories");
    }
    
    private void _saxCategory(Category category) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", category.getId());
        attrs.addCDATAAttribute("name", category.getName());
        attrs.addCDATAAttribute("color", _categoryHelper.getCategoryColor(category).get("main"));
        
        XMLUtils.startElement(contentHandler, "category", attrs);
        category.getTitle().toSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "category");
    }
    
    private void _saxModules(UserIdentity user) throws SAXException
    {
        Set<WorkspaceModule> modulesWithActivities = _projectManager.getUserProjectsModules(user)
                .stream()
                .filter(m -> !m.getAllowedEventTypes().isEmpty())
                .collect(Collectors.toSet());
        
        for (WorkspaceModule module : modulesWithActivities)
        {
            XMLUtils.startElement(contentHandler, "modules");
            _saxModule(module);
            XMLUtils.endElement(contentHandler, "modules");
        }
    }
    
    private void _saxModule(WorkspaceModule module) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", module.getId());
        attrs.addCDATAAttribute("name", module.getModuleName());
        
        XMLUtils.startElement(contentHandler, "module", attrs);
        module.getModuleTitle().toSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "module");
    }
}
