/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.util.Text;

import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.calendars.Calendar.CalendarVisibility;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventJSONHelper;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendar;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarFactory;
import org.ametys.plugins.workspaces.calendars.task.TaskCalendar;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;
import org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO;

/**
 * Calendar DAO
 */
public class CalendarDAO extends AbstractCalendarDAO
{
    /** Avalon Role */
    public static final String ROLE = CalendarDAO.class.getName();

    /** The tasks list JSON helper */
    protected CalendarEventJSONHelper _calendarEventJSONHelper;

    /** The task DAO */
    protected WorkspaceTaskDAO _taskDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _calendarEventJSONHelper = (CalendarEventJSONHelper) manager.lookup(CalendarEventJSONHelper.ROLE);
        _taskDAO = (WorkspaceTaskDAO) manager.lookup(WorkspaceTaskDAO.ROLE);
    }
        
    /**
     * Get calendar info
     * @param calendar The calendar
     * @param recursive True to get data for sub calendars
     * @param includeEvents True to also include child events
     * @param useICSFormat true to use ICS Format for dates
     * @return the calendar data in a map
     */
    public Map<String, Object> getCalendarData(Calendar calendar, boolean recursive, boolean includeEvents, boolean useICSFormat)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("id", calendar.getId());
        result.put("title", Text.unescapeIllegalJcrChars(calendar.getName()));
        result.put("description", calendar.getDescription());
        result.put("templateDesc", calendar.getTemplateDescription());
        result.put("color", calendar.getColor());
        result.put("visibility", calendar.getVisibility().name().toLowerCase());
        result.put("public", calendar.getVisibility() == CalendarVisibility.PUBLIC);
        
        if (calendar instanceof WorkflowAwareCalendar calendarWA)
        {
            result.put("workflowName", calendarWA.getWorkflowName());
        }
        
        result.put("isTaskCalendar", calendar instanceof TaskCalendar);
        result.put("isTaskCalendarDisabled", calendar instanceof TaskCalendar cal && cal.isDisabled());
        
        if (recursive)
        {
            List<Map<String, Object>> calendarList = new LinkedList<>();
            result.put("calendars", calendarList);
            for (Calendar child : calendar.getChildCalendars())
            {
                calendarList.add(getCalendarData(child, recursive, includeEvents, useICSFormat));
            }
        }
        
        if (includeEvents)
        {
            List<Map<String, Object>> eventList = new LinkedList<>();
            result.put("events", eventList);
            
            for (CalendarEvent event : calendar.getAllEvents())
            {
                eventList.add(_calendarEventJSONHelper.eventAsJson(event, false, useICSFormat));
            }
        }

        result.put("rights", _extractCalendarRightData(calendar));
        result.put("token", getCalendarIcsToken(calendar, true));
        
        
        return result;
    }
    
    /**
     * Get calendar info
     * @param calendar The calendar
     * @return the calendar data in a map
     */
    public Map<String, Object> getCalendarProperties(Calendar calendar)
    {
        return getCalendarData(calendar, false, false, false);
    }
    /**
     * Add a calendar
     * @param inputName The desired name for the calendar
     * @param color The calendar color
     * @param isPublic true if the calendar is public
     * @return The result map with id, parentId and name keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addCalendar(String inputName, String color, boolean isPublic)
    {
        String rootId = _getCalendarRoot(true).getId();
        return addCalendar(rootId, inputName, StringUtils.EMPTY, StringUtils.EMPTY, color, isPublic ? CalendarVisibility.PUBLIC.name() : CalendarVisibility.PRIVATE.name(), "calendar-default", false);
    }
    
    /**
     * Add a calendar
     * @param id The identifier of the parent in which the calendar will be added
     * @param inputName The desired name for the calendar
     * @param description The calendar description
     * @param templateDesc The calendar template description
     * @param color The calendar color
     * @param visibility The calendar visibility
     * @param workflowName The calendar workflow name
     * @param renameIfExists True to rename if existing
     * @return The result map with id, parentId and name keys
     */
    public Map<String, Object> addCalendar(String id, String inputName, String description, String templateDesc, String color, String visibility, String workflowName, Boolean renameIfExists)
    {
        return addCalendar(_resolver.resolveById(id), inputName, description, templateDesc, color, visibility, workflowName, renameIfExists, true, true);
    }
        
    /**
     * Add a calendar
     * @param parent The parent in which the calendar will be added
     * @param inputName The desired name for the calendar
     * @param description The calendar description
     * @param templateDesc The calendar template description
     * @param color The calendar color
     * @param visibility The calendar visibility
     * @param workflowName The calendar workflow name
     * @param renameIfExists True to rename if existing
     * @param checkRights true to check if the current user have enough rights to create the calendar
     * @param notify True to notify the calendar creation
     * @return The result map with id, parentId and name keys
     */
    public Map<String, Object> addCalendar(ModifiableTraversableAmetysObject parent, String inputName, String description, String templateDesc, String color, String visibility, String workflowName, Boolean renameIfExists, Boolean checkRights, boolean notify)
    {
        String originalName = Text.escapeIllegalJcrChars(inputName);
        
        // Check user right
        if (checkRights)
        {
            _checkUserRights(parent, RIGHTS_CALENDAR_ADD);
        }
        
        if (BooleanUtils.isNotTrue(renameIfExists) && parent.hasChild(originalName))
        {
            getLogger().warn("Cannot create the calendar with name '" + originalName + "', an object with same name already exists.");
            return Map.of("message", "already-exist");
        }
        
        if (!_explorerResourcesDAO.checkLock(parent))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify the object '" + parent.getName() + "' but it is locked by another user");
            return Map.of("message", "locked");
        }
        
        int index = 2;
        String name = originalName;
        while (parent.hasChild(name))
        {
            name = originalName + " (" + index + ")";
            index++;
        }
        
        JCRCalendar calendar = parent.createChild(name, JCRCalendarFactory.CALENDAR_NODETYPE);
        calendar.setWorkflowName(workflowName);
        calendar.setDescription(description);
        calendar.setTemplateDescription(templateDesc);
        calendar.setColor(color);
        calendar.setVisibility(StringUtils.isNotEmpty(visibility) ? CalendarVisibility.valueOf(visibility.toUpperCase()) : CalendarVisibility.PRIVATE);
        parent.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, calendar.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, calendar.getPath());
        
        if (notify)
        {
            _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_CREATED, _currentUserProvider.getUser(), eventParams));
        }
        
        return getCalendarProperties(calendar);
    }

    /**
     * Edit a calendar
     * @param id The identifier of the calendar
     * @param inputName The new name
     * @param templateDesc The new calendar template description
     * @param color The calendar color
     * @param isPublic true if the calendar is public
     * @return The result map with id and name keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editCalendar(String id, String inputName, String templateDesc, String color, boolean isPublic)
    {
        CalendarVisibility visibility = isPublic ? CalendarVisibility.PUBLIC : CalendarVisibility.PRIVATE;
        
        assert id != null;
        String rename = Text.escapeIllegalJcrChars(inputName);
        
        JCRCalendar calendar = _resolver.resolveById(id);

        _checkUserRights(calendar, RIGHTS_CALENDAR_EDIT);
        
        String name = calendar.getName();
        ModifiableTraversableAmetysObject parent = calendar.getParent();
        
        if (!StringUtils.equals(rename, name) && parent.hasChild(rename))
        {
            getLogger().warn("Cannot edit the calendar with the new name '" + inputName + "', an object with same name already exists.");
            return Map.of("message", "already-exist");
        }
        
        if (!_explorerResourcesDAO.checkLock(calendar))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify calendar '" + calendar.getName() + "' but it is locked by another user");
            return Map.of("message", "locked");
        }
        
        if (!StringUtils.equals(name, rename))
        {
            int index = 2;
            name = Text.escapeIllegalJcrChars(rename);
            while (parent.hasChild(name))
            {
                name = rename + " (" + index + ")";
                index++;
            }
            calendar.rename(name);
        }
        
        calendar.setTemplateDescription(templateDesc);
        calendar.setColor(color);
        calendar.setVisibility(visibility);
        
        parent.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, calendar.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, calendar.getPath());
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_UPDATED, _currentUserProvider.getUser(), eventParams));

        return getCalendarProperties(calendar);
    }
    
    /**
     * Edit the task calendar
     * @param inputName the input name
     * @param color the color
     * @param isPublic <code>true</code> if the calendar is public
     * @param disabled <code>true</code> if the calendar is disabled
     * @return the calendar properties
     * @throws IllegalAccessException if a right error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editTaskCalendar(String inputName, String color, boolean isPublic, boolean disabled) throws IllegalAccessException
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        TaskCalendar taskCalendar = getTaskCalendar(project, false);
        
        // Check user right
        _checkUserRights(_getCalendarRoot(project, false), RIGHTS_CALENDAR_EDIT);
        
        if (taskCalendar != null)
        {
            taskCalendar.rename(inputName);
            taskCalendar.setColor(color);
            taskCalendar.setVisibility(isPublic ? CalendarVisibility.PUBLIC : CalendarVisibility.PRIVATE);
            taskCalendar.disable(disabled);
        }
        return getCalendarProperties(taskCalendar);
    }
    
    /**
     * Delete a calendar
     * @param id The id of the calendar
     * @return The result map with id, parent id and message keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteCalendar(String id)
    {
        Map<String, Object> result = new HashMap<>();

        assert id != null;
        
        JCRCalendar calendar = _resolver.resolveById(id);

        _checkUserRights(calendar, RIGHTS_CALENDAR_DELETE);
        
        if (!_explorerResourcesDAO.checkLock(calendar))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to delete calendar'" + calendar.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        ModifiableExplorerNode parent = calendar.getParent();
        String parentId = parent.getId();
        String name = calendar.getName();
        String path = calendar.getPath();
        
        calendar.remove();
        parent.saveChanges();
     
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, id);
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parentId);
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, path);
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_DELETED, _currentUserProvider.getUser(), eventParams));
        
        result.put("id", id);
        result.put("parentId", parentId);
        
        return result;
    }
        
    /**
     * Get or create the calendar ICS token
     * @param calendar The calendar
     * @param createIfNotExisting Create the token if none exists for the given calendar
     * @return The token
     */
    public String getCalendarIcsToken(Calendar calendar, boolean createIfNotExisting)
    {
        String token = calendar.getIcsUrlToken();
        
        if (createIfNotExisting && token == null && calendar instanceof JCRCalendar)
        {
            token = UUID.randomUUID().toString();
            ((JCRCalendar) calendar).setIcsUrlToken(token);
            ((JCRCalendar) calendar).saveChanges();
        }
        
        return token;
    }
    
    /**
     * Retrieve the calendar for the matching ICS token
     * @param token The ICS token
     * @return The calendar, or null if not found
     */
    public Calendar getCalendarFromIcsToken(String token)
    {
        if (StringUtils.isEmpty(token))
        {
            return null;
        }
        
        Expression expr = new StringExpression(JCRCalendar.CALENDAR_ICS_TOKEN, Operator.EQ, token);
        String calendarsQuery = QueryHelper.getXPathQuery(null, JCRCalendarFactory.CALENDAR_NODETYPE, expr);
        AmetysObjectIterable<JCRCalendar> calendars = _resolver.query(calendarsQuery);
        AmetysObjectIterator<JCRCalendar> calendarsIterator = calendars.iterator();
        
        if (calendarsIterator.getSize() > 0)
        {
            return calendarsIterator.next();
        }
        
        // Don't find a token in default calendars, so check in the task calendars
        return _projectManager.getProjects()
            .stream()
            .map(p -> this.getTaskCalendar(p, true))
            .filter(Objects::nonNull)
            .filter(c -> c.getIcsUrlToken().equals(token))
            .findFirst()
            .orElse(null);
    }
    
    /**
     * Internal method to extract the data concerning the right of the current user for a calendar
     * @param calendar The calendar
     * @return The map of right data. Keys are the rights id, and values indicates whether the current user has the right or not.
     */
    protected  Map<String, Object> _extractCalendarRightData(Calendar calendar)
    {
        Map<String, Object> rightsData = new HashMap<>();
        
        UserIdentity user = _currentUserProvider.getUser();
        boolean isTaskCalendar = calendar instanceof TaskCalendar;
        
        // Add
        rightsData.put("add-event", !isTaskCalendar && _rightManager.hasRight(user, RIGHTS_EVENT_ADD, calendar) == RightResult.RIGHT_ALLOW);
        
        // edit - delete
        rightsData.put("edit", !isTaskCalendar && _rightManager.hasRight(user, RIGHTS_CALENDAR_EDIT, calendar) == RightResult.RIGHT_ALLOW);
        rightsData.put("delete", !isTaskCalendar && _rightManager.hasRight(user, RIGHTS_CALENDAR_DELETE, calendar) == RightResult.RIGHT_ALLOW);
        
        return rightsData;
    }
    
    /**
     * Get the data of every available calendar for the current project
     * @return the list of calendars
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getCalendars()
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        
        _checkReadAccess(project, CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        List<Map<String, Object>> calendarsData = new ArrayList<>();
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        for (Calendar calendar : calendarModule.getCalendars(project, true))
        {
            if (calendarModule.canView(calendar))
            {
                calendarsData.add(this.getCalendarProperties(calendar));
            }
        }
        
        return calendarsData;
    }

    /**
     * Get the colors of calendars
     * @return colors
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, CalendarColorsComponent.CalendarColor> getColors()
    {
        return _calendarColors.getColors();
    }
    
    /**
     * Get user rights on root calendar of current project
     * @return the user rights
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getUserRights()
    {
        Map<String, Object> results = new HashMap<>();
        ModifiableTraversableAmetysObject calendarRoot = _getCalendarRoot(false);
        
        UserIdentity user = _currentUserProvider.getUser();
        results.put("canCreateCalendar", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_CALENDAR_ADD, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canEditCalendar", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_CALENDAR_EDIT, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canRemoveCalendar", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_CALENDAR_DELETE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canCreateEvent", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_EVENT_ADD, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canEditEvent", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_EVENT_EDIT, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canRemoveAnyEvent", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_EVENT_DELETE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canRemoveSelfEvent", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_EVENT_DELETE_OWN, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canHandleResource", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_HANDLE_RESOURCE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canBookResource", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_BOOK_RESOURCE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("sharePrivateCalendar", calendarRoot != null && _rightManager.hasRight(user, RIGHTS_EVENT_EDIT, calendarRoot) == RightResult.RIGHT_ALLOW);
        
        return results;
    }

    /**
     * Get the calendar root
     * @param createIfNotExist true to create root if not exist yet
     * @return the calendar root
     */
    protected ModifiableTraversableAmetysObject _getCalendarRoot(boolean createIfNotExist)
    {
        return _getCalendarRoot(_workspaceHelper.getProjectFromRequest(), createIfNotExist);
    }
    
    /**
     * Get the calendar root form the project
     * @param project the project
     * @param createIfNotExist true to create root if not exist yet
     * @return the calendar root
     */
    protected ModifiableTraversableAmetysObject _getCalendarRoot(Project project, boolean createIfNotExist)
    {
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        return calendarModule.getCalendarsRoot(project, createIfNotExist);
    }
    
    /**
     * Get the data of calendar used to store resources
     * @return the calendar used to store resources
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getResourceCalendar()
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        _checkReadAccess(project, CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        Calendar calendar = calendarModule.getResourceCalendar(project);
        
        return this.getCalendarProperties(calendar);
    }
    
    /**
     * Get the task calendar
     * @param project the project
     * @param onlyIfEnabled <code>true</code> to return the task calendar only if it is enabled
     * @return the task calendar
     */
    public TaskCalendar getTaskCalendar(Project project, boolean onlyIfEnabled)
    {
        if (_projectManager.isModuleActivated(project, TasksWorkspaceModule.TASK_MODULE_ID))
        {
            JCRResourcesCollection root = (JCRResourcesCollection) _getCalendarRoot(project, false);
            TaskCalendar calendar = new TaskCalendar(project, root, _taskDAO);
            return !onlyIfEnabled || !calendar.isDisabled() ? calendar : null;
        }
        
        return null;
    }
    
    /**
     * <code>true</code> if the current user has read access on the task calendar
     * @param project the project
     * @return <code>true</code> if the current user has read access on the task calendar
     */
    public boolean hasTaskCalendarReadAccess(Project project)
    {
        TasksWorkspaceModule taskModule = (TasksWorkspaceModule) _workspaceModuleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        DefaultTraversableAmetysObject tasksRoot = taskModule.getTasksRoot(project, true);
        return _rightManager.currentUserHasReadAccess(tasksRoot);
    }
}
