/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.jcr;

import javax.jcr.Node;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObjectFactory;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link AmetysObjectFactory} for handling {@link JCRCalendar}s.
 */
public class JCRCalendarFactory extends DefaultTraversableAmetysObjectFactory
{
    /** JCR nodetype for resources collection */
    public static final String CALENDAR_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":calendar";

    /** The calendar model */
    private Model _calendarModel;
    
    /** The project manager */
    private ProjectManager _projectManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
    }
    
    @Override
    public JCRCalendar getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new JCRCalendar(node, parentPath, this);
    }

    /**
     * Get the project manager
     * @return the project manager
     */
    public ProjectManager getProjectManager()
    {
        return _projectManager;
    }
    
    /**
     * Get the calendar model
     * @return the calendar model
     */
    public Model getCalendarEventModel()
    {
        if (_calendarModel == null)
        {
            try
            {
                String role = ModelItemTypeExtensionPoint.ROLE_MODEL_AWARE_BASIC;
                _calendarModel = Model.of(
                        "calendar.model.id", 
                        "calendar.model.family.id",

                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_TITLE, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_DESC, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_COLOR, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_VISIBILITY, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_WORKFLOW_NAME, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_TEMPLATE_DESC, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                        DefaultElementDefinition.of(JCRCalendar.CALENDAR_ICS_TOKEN, false, ModelItemTypeConstants.STRING_TYPE_ID, role)
                        
                        );
            }
            catch (Exception e) 
            {
                getLogger().error("An error occurred getting the calendar resource model", e);
                throw new RuntimeException("An error occurred getting the calendar resource model", e);
            }
        }
        return _calendarModel;
    }
}
