/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.task;

import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.ModifiableCalendar;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO;

/**
 * Task calendar object
 */
public class TaskCalendar implements ModifiableCalendar
{
    /** The attribute name to detect is the task calendar is disabled */
    public static final String TASK_CALENDAR_DISABLED_ATTR = "taskCalendarDisabled";
    
    /** The attribute name for the task calendar name */
    public static final String TASK_CALENDAR_NAME_ATTR = "taskCalendarName";
    
    /** The attribute name for the task calendar token */
    public static final String TASK_CALENDAR_TOKEN_ATTR = "taskCalendarToken";
    
    /** The attribute name for the task calendar visibility */
    public static final String TASK_CALENDAR_VISIBILITY_ATTR = "taskCalendarVisibility";
    
    /** The attribute name for the task calendar color */
    public static final String TASK_CALENDAR_COLOR_ATTR = "taskCalendarColor";
    
    
    private WorkspaceTaskDAO _taskDAO;
    private Project _project;
    private JCRResourcesCollection _calendarModuleRoot;
    /**
     * The constructor
     * @param project the project
     * @param calendarModuleRoot the calendar module root
     * @param taskDAO the task DAO
     */
    public TaskCalendar(Project project, JCRResourcesCollection calendarModuleRoot, WorkspaceTaskDAO taskDAO)
    {
        _project = project;
        _taskDAO = taskDAO;
        _calendarModuleRoot = calendarModuleRoot;
    }
    
    /**
     * <code>true</code> if the calendar is disabled
     * @return <code>true</code> if the calendar is disabled
     */
    public boolean isDisabled()
    {
        return _calendarModuleRoot.getValue(TASK_CALENDAR_DISABLED_ATTR, false);
    }
    
    /**
     * Disable the task calendar
     * @param disabled <code>true</code> to disable the task calendar
     */
    public void disable(boolean disabled)
    {
        _calendarModuleRoot.setValue(TASK_CALENDAR_DISABLED_ATTR, disabled);
        _calendarModuleRoot.saveChanges();
    }
    
    public String getId() throws AmetysRepositoryException
    {
        return TaskCalendar.class.getName();
    }
    
    public String getName() throws AmetysRepositoryException
    {
        return _calendarModuleRoot.getValue(TASK_CALENDAR_NAME_ATTR, StringUtils.EMPTY);
    }
    
    public void rename(String name)
    {
        _calendarModuleRoot.setValue(TASK_CALENDAR_NAME_ATTR, name);
        _calendarModuleRoot.saveChanges();        
    }
    
    public Project getProject()
    {
        return _project;
    }
    
    public String getColor()
    {
        return _calendarModuleRoot.getValue(TASK_CALENDAR_COLOR_ATTR, "col1");
    }

    public void setColor(String color)
    {
        _calendarModuleRoot.setValue(TASK_CALENDAR_COLOR_ATTR, color);
        _calendarModuleRoot.saveChanges();
    }
    
    public CalendarVisibility getVisibility()
    {
        return CalendarVisibility.valueOf(_calendarModuleRoot.getValue(TASK_CALENDAR_VISIBILITY_ATTR, CalendarVisibility.PRIVATE.name()));
    }

    public void setVisibility(CalendarVisibility visibility)
    {
        _calendarModuleRoot.setValue(TASK_CALENDAR_VISIBILITY_ATTR, visibility.name());
        _calendarModuleRoot.saveChanges();
    }
    
    public String getIcsUrlToken()
    {
        String token = _calendarModuleRoot.getValue(TASK_CALENDAR_TOKEN_ATTR);
        if (StringUtils.isBlank(token))
        {
            token = UUID.randomUUID().toString();
            setIcsUrlToken(token);
        }
        return token;
    }

    public void setIcsUrlToken(String token)
    {
        _calendarModuleRoot.setValue(TASK_CALENDAR_TOKEN_ATTR, token);
        _calendarModuleRoot.saveChanges();

    }
    
    public String getDescription()
    {
        return null;
    }
    
    public void setDescription(String desc)
    {
        // Do nothing
    }
    
    public String getTemplateDescription()
    {
        return null;
    }
    
    public void setTemplateDescription(String templateDesc)
    {
        // Do nothing
    }

    public List<Calendar> getChildCalendars()
    {
        return List.of();
    }
    
    public List<CalendarEvent> getAllEvents()
    {
        if (isDisabled())
        {
            return List.of();
        }
        
        return _taskDAO.getProjectTasks(_project)
                    .stream()
                    .filter(t -> t.getStartDate() != null || t.getDueDate() != null)
                    .map(t -> new TaskCalendarEvent(this, t))
                    .map(CalendarEvent.class::cast)
                    .toList();
    }
    
    public Map<CalendarEvent, List<CalendarEventOccurrence>> getEvents(ZonedDateTime startDate, ZonedDateTime endDate)
    {
        Map<CalendarEvent, List<CalendarEventOccurrence>> events = new HashMap<>();
        
        for (CalendarEvent event : getAllEvents())
        {
            List<CalendarEventOccurrence> occurences = event.getOccurrences(startDate, endDate);
            if (!occurences.isEmpty())
            {
                events.put(event, occurences);
            }
        }
        
        return events;
    }
}
