/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.workflow;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.plugins.workflow.store.AmetysObjectWorkflowStore;
import org.ametys.plugins.workspaces.calendars.CalendarDAO;
import org.ametys.plugins.workspaces.calendars.ObservationConstants;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventDAO;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventJSONHelper;
import org.ametys.plugins.workspaces.calendars.events.ModifiableCalendarEvent;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendar;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEvent;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEventFactory;
import org.ametys.plugins.workspaces.tags.ProjectTagsDAO;
import org.ametys.plugins.workspaces.workflow.AbstractNodeWorkflowComponent;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.StoreException;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.WorkflowEntry;
import com.opensymphony.workflow.spi.WorkflowStore;

/**
 * Action for adding a calendar event
 */
public class AddEventFunction extends AbstractNodeWorkflowComponent implements EnhancedFunction
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Observer manager. */
    protected ObservationManager _observationManager;
    
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The project tags DAO */
    protected ProjectTagsDAO _projectTagsDAO;

    /** Calendar manager for workspaces */
    protected CalendarDAO _calendarDAO;
    
    /** Calendar event manager for workspaces */
    protected CalendarEventDAO _calendarEventDAO;

    /** The tasks list JSON helper */
    protected CalendarEventJSONHelper _calendarEventJSONHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _projectTagsDAO = (ProjectTagsDAO) smanager.lookup(ProjectTagsDAO.ROLE);
        _calendarDAO = (CalendarDAO) smanager.lookup(CalendarDAO.ROLE);
        _calendarEventDAO = (CalendarEventDAO) smanager.lookup(CalendarEventDAO.ROLE);
        _calendarEventJSONHelper = (CalendarEventJSONHelper) smanager.lookup(CalendarEventJSONHelper.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) transientVars.get("parameters");
        if (jsParameters == null)
        {
            throw new WorkflowException("Missing JS parameters");
        }
        
        ExplorerNode object = getExplorerNode(transientVars);
       
        String parentId = (String) jsParameters.get("parentId");
        
        UserIdentity user = getUser(transientVars);
        String selectedNode = (String) jsParameters.get("selectedNode");
        
        @SuppressWarnings("unchecked")
        Map<String, Object> result = (Map<String, Object>) transientVars.get("result");
        
        // FIXME EXPLORER-441 Remove "renameIfExists" on calendar events workflow
        // boolean renameIfExists = (Boolean) jsParameters.get("renameIfExists");
        assert parentId != null;
        
        if (!(object instanceof JCRCalendar))
        {
            throw new IllegalArgumentException("Cannot add an event on a non-modifiable calendar '" + object.getId() + "'");
        }
        
        JCRCalendar modifiableCalendar = (JCRCalendar) object;

        // Create new event
        JCRCalendarEvent event = modifiableCalendar.createChild("ametys:calendar-event", JCRCalendarEventFactory.CALENDAR_EVENT_NODETYPE);

        ZonedDateTime now = ZonedDateTime.now();
        event.setCreationDate(now);
        event.setLastModified(now);
        event.setCreator(user);

        // Set event's properties
        _setEventData(event, transientVars, jsParameters);
        
        // Bind workflow store
        _initializeWorkflow(event, transientVars, jsParameters);
        
        modifiableCalendar.saveChanges();

        transientVars.put("eventId", event.getId());
        
        result.put("id", event.getId());
        result.put("parentId",  selectedNode == null ? parentId : selectedNode); 
        result.put("event", _calendarEventJSONHelper.eventAsJsonWithOccurrences(event, false, event.getStartDate(), event.getEndDate()));
        // Notify listeners
        _notifyListeners(event);
    }
    
    /**
     * Initialize the workflow store
     * @param event The event
     * @param transientVars The transient variables
     * @param jsParameters The JS parameters
     * @throws WorkflowException if an error occurred
     */
    protected void _initializeWorkflow (JCRCalendarEvent event, Map transientVars, Map<String, Object> jsParameters) throws WorkflowException
    {
        long workflowId = ((WorkflowEntry) transientVars.get("entry")).getId();
        
        try
        {
            event.setWorkflowId(workflowId);
            WorkflowStore workflowStore = (WorkflowStore) transientVars.get("store");
            
            if (workflowStore instanceof AmetysObjectWorkflowStore)
            {
                AmetysObjectWorkflowStore ametysObjectWorkflowStore = (AmetysObjectWorkflowStore) workflowStore;
                ametysObjectWorkflowStore.bindAmetysObject(event);
            }
        }
        catch (StoreException e)
        {
            throw new WorkflowException("Unable to link the workflow to the content", e);
        }
    }
    
    /**
     * Set the event data
     * @param event The event
     * @param transientVars The transient variables
     * @param jsParameters The JS parameters
     * @throws WorkflowException if an error occurred
     */
    protected void _setEventData (ModifiableCalendarEvent event, Map transientVars, Map<String, Object> jsParameters) throws WorkflowException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> result = (Map<String, Object>) transientVars.get("result");
        String title = (String) jsParameters.get("title");
        String desc = (String) jsParameters.get("description");
        String startDateAsString = (String) jsParameters.get("startDate");

        String zoneId = (String) jsParameters.get("zoneId");
        ZoneId zone = ZoneId.of(zoneId);
        event.setZone(zone);
        
        ZonedDateTime startDate = DateUtils.parseZonedDateTime(startDateAsString).withZoneSameInstant(zone);
        String endDateAsString = (String) jsParameters.get("endDate");
        ZonedDateTime endDate = DateUtils.parseZonedDateTime(endDateAsString).withZoneSameInstant(zone);
        
        Object rawFullDay = jsParameters.get("fullDay");
        Boolean fullDay = rawFullDay instanceof Boolean ? (Boolean) rawFullDay : Boolean.parseBoolean((String) rawFullDay);
        
        String recurrenceType = (String) jsParameters.get("recurrenceType");
        String untilDateAsString = (String) jsParameters.get("untilDate");
        ZonedDateTime untilDate = DateUtils.parseZonedDateTime(untilDateAsString);
        
        UserIdentity user = getUser(transientVars);
        
        event.setTitle(title);
        event.setDescription(desc);
        event.setFullDay(fullDay);
        
        event.setStartDate(startDate);
        event.setLastContributor(user);
        event.setLastModified(ZonedDateTime.now());
        
        event.setRecurrenceType(recurrenceType);
        if (untilDate !=  null)
        {
            event.setRepeatUntil(untilDate);
        }
              
        event.setEndDate(endDate);
        
        String location = (String) jsParameters.get("location");
        if (StringUtils.isNotEmpty(location))
        {
            event.setLocation(location);
        }
        
        @SuppressWarnings("unchecked")
        List<Object> tags = (List<Object>) jsParameters.getOrDefault("tags", new ArrayList<>());
        List<String> createdTags = new ArrayList<>();
        List<Map<String, Object>> createdTagsJson = new ArrayList<>();
        // Tag new tags
        for (Object tag : tags)
        {
            // Tag doesn't exist so create the tag
            if (tag instanceof Map)
            {
                @SuppressWarnings("unchecked")
                String tagText = (String) ((Map<String, Object>) tag).get("text");
                List<Map<String, Object>> newTags = _projectTagsDAO.addTags(new String[] {tagText});
                String newTag = (String) newTags.get(0).get("name");
                event.tag(newTag);
                createdTags.add(newTag);
                createdTagsJson.addAll(newTags);
            }
            else
            {
                event.tag((String) tag);
            }
        }
        // Untag unused tags
        for (String tag : event.getTags())
        {
            if (!tags.contains(tag) && !createdTags.contains(tag))
            {
                event.untag(tag);
            }
        }

        @SuppressWarnings("unchecked")
        List<String> resources = (List<String>) jsParameters.getOrDefault("resourceIds", new ArrayList<>());
        event.setResources(resources);
        result.put("newTags", createdTagsJson);
    }
    
    /**
     * Notify listeners that the event has been created
     * @param event The created event
     */
    protected void _notifyListeners (CalendarEvent event)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CALENDAR, event.getCalendar());
        eventParams.put(ObservationConstants.ARGS_CALENDAR_EVENT, event);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, event.getId());
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_PARENT_ID, event.getParent().getId());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CALENDAR_EVENT_CREATED, _currentUserProvider.getUser(), eventParams));
    }

    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.PRE;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_ADD_EVENT_FUNCTION_LABEL");
    }
}
