/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.comments;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.web.URIPrefixHandler;

/**
 * Comments DAO for workspaces
 *
 */
public class CommentsDAO extends org.ametys.web.repository.comment.CommentsDAO
{
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    /** The URI prefix handler */
    protected URIPrefixHandler _prefixHandler;
    private WorkspacesHelper _workspaceHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _populationContextHelper = (PopulationContextHelper) smanager.lookup(PopulationContextHelper.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
        _prefixHandler = (URIPrefixHandler) smanager.lookup(URIPrefixHandler.ROLE);
        _workspaceHelper = (WorkspacesHelper) smanager.lookup(WorkspacesHelper.ROLE);
    }
    
    @Override
    protected void saxCommentAdditionalProperties(ContentHandler contentHandler, Comment comment, int level, Map<String, Object> contextualParameters) throws SAXException
    {
        super.saxCommentAdditionalProperties(contentHandler, comment, level, contextualParameters);
        
        String lang = (String) contextualParameters.get("sitemapLanguage");
        
        Optional<UserIdentity> optAuthorIdentity = getAuthorIdentity(comment, contextualParameters);
        if (optAuthorIdentity.isPresent())
        {
            UserIdentity authorIdentity = optAuthorIdentity.get();
            
            // SAX additionnal user information
            Content userContent = _userDirectoryHelper.getUserContent(authorIdentity, lang);
            if (userContent != null)
            {
                _saxUserImage(contentHandler, authorIdentity, userContent.getLanguage());
                _saxAdditionalUserProperties(contentHandler, userContent);
            }
            else
            {
                _saxUserImage(contentHandler, authorIdentity, lang);
            }
        }
    }
    
    @Override
    protected Map<String, Object> getCommentAuthor(Comment comment, Map<String, Object> contextualParameters)
    {
        Map<String, Object> comment2json = super.getCommentAuthor(comment, contextualParameters);
        
        Optional<UserIdentity> authorIdentity = getAuthorIdentity(comment, contextualParameters);
        if (authorIdentity.isPresent())
        {
            String lang = (String) contextualParameters.get("sitemapLanguage");
            Content userContent = _userDirectoryHelper.getUserContent(authorIdentity.get(), lang);
            if (userContent != null)
            {
                if (userContent.hasDefinition("function"))
                {
                    String function = userContent.getValue("function", false, null);
                    if (StringUtils.isNotEmpty(function))
                    {
                        comment2json.put("author-function", function);
                    }
                }
                
                if (userContent.hasDefinition("organisation"))
                {
                    String organisation = userContent.getValue("organisation", false, null);
                    if (StringUtils.isNotEmpty(organisation))
                    {
                        comment2json.put("author-organisation", organisation);
                    }
                }
                
                if (userContent.hasDefinition("organisation-accronym"))
                {
                    String organisationAccr = userContent.getValue("organisation-accronym", false, null);
                    if (StringUtils.isNotEmpty(organisationAccr))
                    {
                        comment2json.put("author-organisation-accronym", organisationAccr);
                    }
                }
            }
        }
        
        return comment2json;
    }
    
    /**
     * Sax the user avatar
     * @param contentHandler the content handler
     * @param userIdentity the user identity
     * @param language the language code (if image will be taken from user content)
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxUserImage(ContentHandler contentHandler, UserIdentity userIdentity, String language) throws SAXException
    {
        XMLUtils.createElement(contentHandler, "author-img-url", _workspaceHelper.getAvatar(userIdentity, language, 45));
    }
    
    /**
     * Add additional properties for comment's author
     * @param contentHandler the content handler
     * @param userContent the user content
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxAdditionalUserProperties(ContentHandler contentHandler, Content userContent) throws SAXException
    {
        if (userContent.hasDefinition("function"))
        {
            String function = userContent.getValue("function", false, null);
            if (StringUtils.isNotEmpty(function))
            {
                XMLUtils.createElement(contentHandler, "author-function", function);
            }
        }
        
        if (userContent.hasDefinition("organisation"))
        {
            String organisation = userContent.getValue("organisation", false, null);
            if (StringUtils.isNotEmpty(organisation))
            {
                XMLUtils.createElement(contentHandler, "author-organisation", organisation);
            }
        }
        
        if (userContent.hasDefinition("organisation-accronym"))
        {
            String organisationAccr = userContent.getValue("organisation-accronym", false, null);
            if (StringUtils.isNotEmpty(organisationAccr))
            {
                XMLUtils.createElement(contentHandler, "author-organisation-accronym", organisationAccr);
            }
        }
    }
    
    /**
     * Retrieves the {@link UserIdentity} of the comment's author
     * @param comment the comment
     * @param contextualParameters the contextual parameters
     * @return the comment's author identity
     */
    protected Optional<UserIdentity> getAuthorIdentity(Comment comment, Map<String, Object> contextualParameters)
    {
        UserIdentity authorIdentity = comment.getAuthor();
        return authorIdentity != null
                ? Optional.of(authorIdentity)
                : getUserByEmail(comment.getAuthorEmail(), contextualParameters)
                        .map(User::getIdentity);
    }
}
