/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.datapolicy;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.repository.ReactionableObject.ReactionType;
import org.ametys.cms.repository.ReactionableObjectHelper;
import org.ametys.cms.repository.comment.AbstractComment;
import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.PersonalDataProcessingException;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract data policy for anonymization of comments and deletion reactions made by an unknown user in workspaces modules
 */
public abstract class AbstractCommentAndReactionDataPolicy extends AbstractLogEnabled implements PersonalDataPolicy, Serviceable
{
    /** The user manager */
    protected UserManager _userManager;
    /** The repository provider */
    protected Repository _repository;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    private ModelItemTypeExtensionPoint _unversionedDataTypeExtensionPoint;
    private Period _retentionPeriod;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _unversionedDataTypeExtensionPoint = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_UNVERSIONED);
        
        Long config = Config.getInstance().<Long>getValue("cms.comment.data.policy.retention.period", false, null);
        // disable the processing if config is blank
        _retentionPeriod = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    
    public AnonymizationResult process(UserStatusInfo userStatusInfo) throws PersonalDataProcessingException
    {
        if (_retentionPeriod == null
            || userStatusInfo.getMissingSinceDate().isAfter(ZonedDateTime.now().minus(_retentionPeriod)))
        {
            return AnonymizationResult.TOO_EARLY;
        }
        else
        {
            int handled = 0;
            boolean error = false;
            Session session = null;
            try
            {
                session = _repository.login();
                QueryManager queryManager = session.getWorkspace().getQueryManager();
                HandlingResult result = handleComments(userStatusInfo, queryManager);
                handled += result.handled();
                error |= result.failed() > 0;
                handled += handleCommentReactions(userStatusInfo, queryManager);
                
                return error ? AnonymizationResult.ERROR : handled == 0 ? AnonymizationResult.NO_DATA : AnonymizationResult.PROCESSED;
            }
            catch (RepositoryException e)
            {
                throw new PersonalDataProcessingException("An error prevented the processing of comment from '" + userStatusInfo.getUserIdentity() + "'.", e);
            }
            finally
            {
                if (session != null)
                {
                    session.logout();
                }
            }
        }
    }
    
    /**
     * Handle anonymization of comments
     * @param userStatusInfo The user status infos
     * @param queryManager the query manager
     * @return the result
     * @throws RepositoryException if an error occurred
     */
    protected HandlingResult handleComments(UserStatusInfo userStatusInfo, QueryManager queryManager) throws RepositoryException
    {
        int handled = 0;
        int failed = 0;
        
        String query = getCommentsQuery(userStatusInfo);
        
        @SuppressWarnings("deprecation")
        QueryResult result = queryManager.createQuery(query.toString(), Query.XPATH).execute();
        NodeIterator nodes = result.getNodes();
        while (nodes.hasNext())
        {
            Node commentNode = nodes.nextNode();
            if (handleComment(commentNode))
            {
                handled++;
            }
            else
            {
                failed++;
            }
        }
        
        if (handled > 0 || failed > 0)
        {
            ForensicLogger.info("data.policy.gdpr.anonymize." + getLogCategory() + ".comments", Map.of("handled", Long.toString(handled), "failed", Long.toString(failed), "identity", userStatusInfo.getUserIdentity()), UserPopulationDAO.SYSTEM_USER_IDENTITY);
        }
        return new HandlingResult(handled, failed);
    }
    
    /**
     * Get the log category describing the object
     * @return the log category
     */
    protected abstract String getLogCategory();

    /**
     * Get the query to retrieving comments
     * @param userStatusInfo the user status info
     * @return the JCR query
     */
    protected String getCommentsQuery(UserStatusInfo userStatusInfo)
    {
        StringBuilder query = new StringBuilder("//element(*, " + getObjectPrimaryType() + ")")
                .append("//ametys:comments")
                .append("/*[").append(new UserExpression(AbstractComment.METADATA_COMMENT_AUTHOR, Operator.EQ, userStatusInfo.getUserIdentity()).build()).append("]");
        
        return query.toString();
    }
    
    /**
     * Get the primary type of object that holds comments
     * @return the primary type
     */
    protected abstract String getObjectPrimaryType();
    
    /**
     * Handle anonymization of comment
     * @param commentNode the comment node
     * @return true if anomymization succeed, false otherwise
     */
    protected abstract boolean handleComment(Node commentNode);
    
    /**
     * Get parent holder node and comment id from comment node
     * @param commentNode the comment node
     * @return the parent node and comment id or null if not found
     */
    protected Pair<Node, String> getObjectNodeAndCommentId(Node commentNode)
    {
        String commentNodePath = null;
        try
        {
            commentNodePath = commentNode.getPath();
            
            // Retrieve the comment id by going up the hierarchy
            // …/ametys-internal:resources/forums/thread-ID/ametys:comments/ametys:comment-x/ametys:comments/ametys:comment-y
            // …/ametys-internal:resources/tasks/tasks-root/task-ID/ametys:comments/ametys:comments/ametys:comment-x/ametys:comments/ametys:comment-y
            // => comment-x_comment-y
            
            // We start at the comment node
            StringBuilder commentId = new StringBuilder(StringUtils.removeStart(commentNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":"));
            
            // Go up to parent object while reconstructing the ID of comment
            Node parentNode = commentNode.getParent();
            while (parentNode != null && !getObjectPrimaryType().equals(parentNode.getPrimaryNodeType().getName()))
            {
                if (parentNode.getName().startsWith(RepositoryConstants.NAMESPACE_PREFIX + ":" + AbstractComment.COMMENT_NAME_PREFIX))
                {
                    commentId
                        .insert(0, AbstractComment.ID_SEPARATOR)
                        .insert(0, StringUtils.removeStart(parentNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":"));
                }
                
                parentNode = parentNode.getParent(); // continue iterating
            }
            
            if (parentNode == null)
            {
                getLogger().error("Parent holder object not found from comment node '{}'", commentNodePath);
                return null;
            }
            
            return Pair.of(parentNode, commentId.toString());
        }
        catch (RepositoryException | AmetysRepositoryException e)
        {
            getLogger().error("Failed to retrieve parent holder and comment from comment node '{}'", commentNodePath,  e);
            return null;
        }
    }
    /**
     * Handle reactions
     * @param userStatusInfo the user status infos
     * @param queryManager the query manager
     * @return the number of handled reactions
     * @throws RepositoryException if an error occurred
     */
    protected int handleCommentReactions(UserStatusInfo userStatusInfo, QueryManager queryManager) throws RepositoryException
    {
        int handled = 0;
        for (ReactionType type : ReactionType.values())
        {
            // Find any node with the reaction type name under root comments of an object with the user as issuer
            StringBuilder query = new StringBuilder("//element(*, " + getObjectPrimaryType() + ")/ametys:comments")
                    .append("//").append(RepositoryConstants.NAMESPACE_PREFIX).append(":").append(type.name().toLowerCase())
                    .append("[").append(new UserExpression("users", Operator.EQ, userStatusInfo.getUserIdentity(), true).build()).append("]");
            
            @SuppressWarnings("deprecation")
            QueryResult result = queryManager.createQuery(query.toString(), Query.XPATH).execute();
            NodeIterator nodes = result.getNodes();
            while (nodes.hasNext())
            {
                // We get the parent reactionable object
                Node parentNode = nodes.nextNode().getParent();
                JCRRepositoryData repositoryData = new JCRRepositoryData(parentNode);
                ModifiableModelLessDataHolder reactionableObject = new DefaultModifiableModelLessDataHolder(_unversionedDataTypeExtensionPoint, repositoryData);
                ReactionableObjectHelper.removeReaction(reactionableObject, userStatusInfo.getUserIdentity(), type);
                
                parentNode.getSession().save();
                handled++;
            }
        }
        
        if (handled > 0)
        {
            ForensicLogger.info("data.policy.gdpr.remove." + getLogCategory() + ".reactions", Map.of("handled", Long.toString(handled), "identity", userStatusInfo.getUserIdentity()), UserPopulationDAO.SYSTEM_USER_IDENTITY);
        }
        
        return handled;
    }
    
    /**
     * Record for processing result
     * @param handled the number of elements handled
     * @param failed the number of elements in error
     */
    protected static record HandlingResult(int handled, int failed) { }
}
