/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.documents.onlyoffice;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO;

/**
 * Action to handle OnlyOffice communications
 */
public class GetOnlyOfficeResponse extends ServiceableAction implements ThreadSafe
{
    /**
     * Enumeration for user connection status    
     */
    public enum UserStatus 
    {
        /** Unknown */
        UNKNOWN,
        /** User connection */
        CONNECTION,
        /** User disconnection */
        DISCONNECTION
    }

    /** The logger */
    protected Logger _logger = LoggerFactory.getLogger(GetOnlyOfficeResponse.class.getName());
    
    /**  JSON utils */
    protected JSONUtils _jsonUtils;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Workspace explorer resource DAO */
    protected WorkspaceExplorerResourceDAO _workspaceExplorerResourceDAO;

    /** The only office manager */
    protected OnlyOfficeKeyManager _onlyOfficeManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _workspaceExplorerResourceDAO = (WorkspaceExplorerResourceDAO) smanager.lookup(WorkspaceExplorerResourceDAO.ROLE);
        _onlyOfficeManager = (OnlyOfficeKeyManager) smanager.lookup(OnlyOfficeKeyManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String resourceId = request.getParameter("id");
        
        HttpServletRequest httpRequest = (HttpServletRequest) objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);
        
        try (InputStream is = httpRequest.getInputStream())
        {
            String body = IOUtils.toString(is, StandardCharsets.UTF_8);
            Map<String, Object> jsonMap = _jsonUtils.convertJsonToMap(body);
            boolean success = callbackHandler(jsonMap, resourceId);
            
            // Expected response from the document storage service is { "error": 0 }, otherwise the document editor will display an error message
            result.put("error", success ? 0 : 1);
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Analyze the response from document editing service on OnlyOffice Server
     * See https://api.onlyoffice.com/editors/callback
     * @param response the response as JSON object
     * @param resourceId id of edited resource
     * @return true if the response is ok and the needed process successed.
     */
    protected boolean callbackHandler(Map<String, Object> response, String resourceId)
    {
        // edited document identifier
        String key = (String) response.get("key");
        if (key == null)
        {
            _logger.error("The edited document identifier is missing in document editing service's response");
            return false;
        }
        
        // status of document
        Object oStatus = response.get("status");
        if (oStatus == null)
        {
            _logger.error("The status of the edited document is missing in document editing service's response");
            return false;
        }
        
        int status = (Integer) oStatus;
        
        _logger.debug("Receive status {} for key '{}' and resource '{}' from document editing service", status, key, resourceId);
        
        switch (status)
        {
            case 0:
                // document not found ?? (0 is not part of documented status)
                return true;
            case 1:
                // document is being edited
                return getUserConnectionStatus(response) != UserStatus.UNKNOWN;
            case 2:
                // document is ready for saving
                // Status 2 is received 10s after the document is closed for editing with the identifier of the user who was the last to send the changes to the document editing service
                try
                {
                    Boolean notModified = (Boolean) response.get("notmodified");
                    
                    if (notModified == null || !notModified)
                    {
                        return saveDocument(response, resourceId);
                    }
                    
                    return true;
                }
                finally
                {
                    _onlyOfficeManager.removeKey(resourceId);
                }
            case 3:
                // document saving error has occurred
                _logger.error("Document editing service returned that document saving error has occurred (3)");
                _onlyOfficeManager.removeKey(resourceId);
                return false;
            case 4:
                // document is closed with no changes
                _logger.info("Document editing service returned that document was closed with no changes (4)");
                _onlyOfficeManager.removeKey(resourceId);
                return getUserConnectionStatus(response) != UserStatus.UNKNOWN;
            case 6:
                // document is being edited, but the current document state is saved (force save)
                return saveDocument(response, resourceId);
            case 7:
                // error has occurred while force saving the document
                _logger.error("Document editing service returned that an error has occurred while force saving the document (7)");
                return false;
            default:
                _logger.error("Document editing service returned an invalid status: " + status);
                return false;
        }
    }
    
    /**
     * Get the status of a user connection
     * @param response response of OO server as JSON
     * @return the user status
     */
    protected UserStatus getUserConnectionStatus(Map<String, Object> response)
    {
        if (response != null && response.containsKey("actions"))
        {
            @SuppressWarnings("unchecked")
            List<Map<String, Object>> actions = (List<Map<String, Object>>) response.get("actions");
            
            for (Map<String, Object> action : actions)
            {
                int type = (Integer) action.get("type");
                return type == 1 ? UserStatus.CONNECTION : UserStatus.DISCONNECTION;
            }
        }

        return UserStatus.UNKNOWN;
    }
    
    /**
     * Save the document if needed
     * @param response the OO response as JSON
     * @param resourceId id of document to edit
     * @return true if save success, false otherwise
     */
    protected boolean saveDocument(Map<String, Object> response, String resourceId)
    {
        if (response.get("notmodified") != null && (boolean) response.get("notmodified"))
        {
            // No save is needed
            _logger.debug("No modification was detected by document editing service");
            return true;
        }
        
        String url = (String) response.get("url");
        if (StringUtils.isEmpty(url))
        {
            _logger.error("Unable to save document, the document url is missing");
            return false;
        }
        
        return processSave(url, resourceId);
    }

    /**
     * Save the document
     * @param documentURI the URI of document to save
     * @param resourceId id of the resource to update
     * @return true if success to save document, false otherwise
     */
    protected boolean processSave(String documentURI, String resourceId)
    {
        try
        {
            URL url = new URL(documentURI);
            
            Resource resource = _resolver.resolveById(resourceId);
            if (_rightManager.hasRight(_currentUserProvider.getUser(), ExplorerResourcesDAO.RIGHTS_RESOURCE_ADD, resource.getParent()) != RightResult.RIGHT_ALLOW)
            {
                _logger.error("User '" + _currentUserProvider.getUser() + "' tried to add file without convenient right [" + ExplorerResourcesDAO.RIGHTS_RESOURCE_ADD + "]");
                return false;
            }
            
            try (InputStream is = url.openStream())
            {
                // Update the document
                Map<String, Object> result = _workspaceExplorerResourceDAO.addFile(is, resource.getName(), resource.getParent().getId(), false /* unarchive */, false /* allow rename */, true /* allow update */);
                if (result != null && result.containsKey("resources"))
                {
                    return true;
                }
                else
                {
                    _logger.error("Failed to update document with id '{}", resourceId);
                    return false;
                }
            }
        }
        catch (IOException e)
        {
            _logger.error("Unable to get document at uri {}", documentURI, e);
            return false;
        }
    }
}
