/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.initialization;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.jackrabbit.value.StringValue;

import org.ametys.cms.rights.Content2ContentTypeRightContextConvertor;
import org.ametys.core.observation.Event;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.userdirectory.page.VirtualUserDirectoryPageFactory;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableSitemapElement;
import org.ametys.web.repository.page.Page;

/**
 * Configure a user directory root page during initialization.
 */
public class UserDirectoryPageInitializer extends DefaultStaticPageInitializer
{
    /** the profile assignment storage EP */
    protected ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _profileAssignmentStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
    }
    
    @Override
    public Optional< ? extends Page> createPage(ModifiableSitemapElement parent)
    {
        Optional< ? extends Page> result = super.createPage(parent);
        if (result.isPresent() && result.get() instanceof ModifiablePage page)
        {
            try
            {
                // FIXME isn't there a better way to set the user directory root ?
                if (page instanceof JCRAmetysObject jcrPage)
                {
                    StringValue virtualUserDirectoryPageFactoryClassName = new StringValue(VirtualUserDirectoryPageFactory.class.getName());
                    jcrPage.getNode().setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, new Value[] {virtualUserDirectoryPageFactoryClassName});
                }
                
                page.setValue(UserDirectoryPageHandler.CONTENT_TYPE_DATA_NAME, WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
                page.setValue(UserDirectoryPageHandler.USER_VIEW_NAME, "main");
                page.setValue(UserDirectoryPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME, "lastname");
                page.setValue(UserDirectoryPageHandler.DEPTH_DATA_NAME, 2);
                
                page.saveChanges();
                
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
                _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_PAGE_CHANGED, _currentUserProvider.getUser(), eventParams));
                
                eventParams.put(org.ametys.plugins.userdirectory.observation.ObservationConstants.ARGS_USER_CONTENT_VIEW_UPDATED, true);
                _observationManager.notify(new Event(org.ametys.plugins.userdirectory.observation.ObservationConstants.EVENT_USER_DIRECTORY_ROOT_UPDATED, _currentUserProvider.getUser(), eventParams));
                
                // Set read right on member content type for any connected user
                String contentTypeContext = Content2ContentTypeRightContextConvertor.CONTENT_TYPE_RIGHT_CONTEXT_PREFIX + "/" + page.getSiteName() + "/" + WorkspacesConstants.MEMBER_CONTENT_TYPE_ID;
                _profileAssignmentStorageEP.allowProfileToAnyConnectedUser(RightManager.READER_PROFILE_ID, contentTypeContext);
                
                eventParams = new HashMap<>();
                eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, contentTypeContext);
                eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, Set.of(RightManager.READER_PROFILE_ID));
                
                _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, _currentUserProvider.getUser(), eventParams));
            }
            catch (RepositoryException e)
            {
                getLogger().error("Failed to set user directory root property", e);
            }
        }
        return result;
    }
}
