/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.keywords;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.AbstractTagProvider;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.DefaultTag;
import org.ametys.cms.tag.TagTargetType;
import org.ametys.cms.tag.TagTargetTypeExtensionPoint;

/**
 * Class representing categories as CMS tag provider.
 * This directly use {@link KeywordJCRProvider} and transform the categories in {@link CMSTag}
 */
public class KeywordTagProvider extends AbstractTagProvider<CMSTag> implements Serviceable
{
    /** Target types */
    protected TagTargetTypeExtensionPoint _targetTypeEP;
    
    /** Category tag provider */
    protected KeywordProviderExtensionPoint _keywordProviderEP;
    
    /** The JCR category provider */
    protected KeywordJCRProvider _jcrKeywordProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _targetTypeEP = (TagTargetTypeExtensionPoint) smanager.lookup(TagTargetTypeExtensionPoint.ROLE);
        _keywordProviderEP = (KeywordProviderExtensionPoint) smanager.lookup(KeywordProviderExtensionPoint.ROLE);
        _jcrKeywordProvider = (KeywordJCRProvider) _keywordProviderEP.getExtension(KeywordJCRProvider.class.getName());
    }
    
    public Map<String, CMSTag> getTags(Map<String, Object> contextualParameters)
    {
        Map<String, CMSTag> cmsTags = new HashMap<>();
        
        Map<String, DefaultTag> keywords = _jcrKeywordProvider.getTags(contextualParameters);
        
        for (Entry<String, DefaultTag> category : keywords.entrySet())
        {
            KeywordCMSTag keywordTag = _transformToCMSTag(category.getValue(), contextualParameters);
            cmsTags.put(keywordTag.getId(), _transformToCMSTag(category.getValue(), contextualParameters));
        }
        
        return cmsTags;
    }
    
    public CMSTag getTag(String tagName, Map<String, Object> contextualParameters)
    {
        String keywordName = tagName;
        if (keywordName.startsWith(KeywordCMSTag.TAG_PREFIX))
        {
            keywordName = StringUtils.substringAfter(tagName, KeywordCMSTag.TAG_PREFIX);
            return _transformToCMSTag(_jcrKeywordProvider.getTag(keywordName, contextualParameters), contextualParameters);
        }
        return null;
    }
    
    public Collection<CMSTag> getTags(String tagName, Map<String, Object> contextualParameters)
    {
        String keywordName = tagName;
        if (keywordName.startsWith(KeywordCMSTag.TAG_PREFIX))
        {
            keywordName = StringUtils.substringAfter(tagName, KeywordCMSTag.TAG_PREFIX);
            Collection<DefaultTag> keywords = _jcrKeywordProvider.getTags(keywordName, contextualParameters);
            
            return keywords.stream()
                    .map(tag -> _transformToCMSTag(tag, contextualParameters))
                    .collect(Collectors.toList());
        }
        return List.of();
    }
    
    public boolean hasTag(String tagName, Map<String, Object> contextualParameters)
    {
        String keywordName = tagName;
        if (keywordName.startsWith(KeywordCMSTag.TAG_PREFIX))
        {
            keywordName = StringUtils.substringAfter(tagName, KeywordCMSTag.TAG_PREFIX);
            return _jcrKeywordProvider.hasTag(keywordName, contextualParameters);
        }
        return false;
    }
    
    /**
     * Transform a keyword in a {@link KeywordCMSTag}, forcing visibility to public and target to CONTENT
     * @param keyword the keyword to transform
     * @param contextualParameters the contextual parameters
     * @return a {@link KeywordCMSTag} with the same values
     */
    protected KeywordCMSTag _transformToCMSTag(DefaultTag keyword, Map<String, Object> contextualParameters)
    {
        if (keyword != null)
        {
            CMSTag parentTag = null;
            DefaultTag parentKeyword = keyword.getParent();
            if (parentKeyword != null)
            {
                parentTag = this.getTag(parentKeyword.getName(), contextualParameters);
            }
            
            TagTargetType targetType = _targetTypeEP.getTagTargetType("CONTENT");
            
            return new KeywordCMSTag(keyword, parentTag, targetType);
        }
        else
        {
            return null;
        }
    }
}
