/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members.observers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Abstract observer for member update
 *
 */
public abstract class AbstractMemberObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    /** The service manager */
    protected ServiceManager _smanager;
    /** The group manager */
    protected GroupManager _groupManager;
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    /** The site manager */
    protected SiteManager _siteManager;
    /** The project manager */
    protected ProjectManager _projectManager;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _smanager = smanager;
        _userDirectoryHelper = (UserDirectoryHelper) smanager.lookup(UserDirectoryHelper.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
    }

    public boolean supports(Event event)
    {
        String eventId = event.getId();
        return eventId.equals(ObservationConstants.EVENT_MEMBER_ADDED) || eventId.equals(ObservationConstants.EVENT_MEMBER_DELETED);
    }

    public int getPriority()
    {
        return MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        // Lazy Loading
        if (_groupManager == null)
        {
            _groupManager = (GroupManager) _smanager.lookup(GroupManager.ROLE);
        }
        
        Map<String, Object> args = event.getArguments();

        MemberType type = (MemberType) args.get(ObservationConstants.ARGS_MEMBER_IDENTITY_TYPE);
        String identity = (String) args.get(ObservationConstants.ARGS_MEMBER_IDENTITY);
        
        Set<UserIdentity> users = null;
        
        if (MemberType.USER == type)
        {
            UserIdentity user = UserIdentity.stringToUserIdentity(identity);
            users = Collections.singleton(user);
        }
        else
        {
            // Group
            Group group = _groupManager.getGroup(GroupIdentity.stringToGroupIdentity(identity));
            users = group.getUsers();
        }
        
        List<Content> userContents = getUserContents(users);
        _internalObserve(event, userContents);
    }
    
    /**
     * Observes the event with access to the user contents
     * @param event the event.
     * @param userContents the user contents
     * @throws Exception if an error occurs.
     */
    protected abstract void _internalObserve(Event event, List<Content> userContents) throws Exception;
    
    /**
     * Get the user contents associated to user identities
     * @param users the users
     * @return the user contents
     */
    protected List<Content> getUserContents(Set<UserIdentity> users)
    {
        String catalogSiteName = _projectManager.getCatalogSiteName();
        
        List<Content> contents = new ArrayList<>();
        
        Site site = _siteManager.getSite(catalogSiteName);
        if (site != null)
        {
            Set<String> languages = site.getSitemaps().stream()
                .map(Sitemap::getName)
                .collect(Collectors.toSet());
            
            for (UserIdentity user : users)
            {
                for (String lang : languages)
                {
                    Content userContent = _userDirectoryHelper.getUserContent(user, lang);
                    if (userContent != null)
                    {
                        contents.add(userContent);
                    }
                }
            }
        }
        
        return contents;
    }
}
