/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.minisite;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.search.query.SiteQuery;
import org.ametys.web.workflow.CreateContentFunction;

import com.google.common.collect.ImmutableSet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Workspaces module for editorial pages
 */
public class MiniSiteWorkspaceModule extends AbstractWorkspaceModule
{
    /** Avalon ROLE */
    public static final String MINISITE_MODULE_ID = MiniSiteWorkspaceModule.class.getName();
    
    /** Workspaces tasks list node name */
    private static final String __WORKSPACES_MINISITE_NODE_NAME = "minisite";
    
    private static final int __INITIAL_WORKFLOW_ACTION_ID = 13;
    
    private static final String __ARTICLE_NUMBER_HEADER_ID = __WORKSPACES_MINISITE_NODE_NAME + "$article_number";
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;

    /** The content searcher factory */
    protected ContentSearcherFactory _contentSearcherFactory;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
    }
    
    @Override
    public String getId()
    {
        return MINISITE_MODULE_ID;
    }
    
    public boolean isUnactivatedByDefault()
    {
        return true;
    }
    
    @Override
    public String getModuleName()
    {
        return __WORKSPACES_MINISITE_NODE_NAME;
    }
    
    public int getOrder()
    {
        return ORDER_MINISITE;
    }
    
    @Override
    protected String getModulePageName()
    {
        return "minisite";
    }
    
    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_MINISITE_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_MINISITE_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_MINISITE_TITLE");
    }
    
    /**
     * Get the name of the first section of minisite
     * @return the section's name
     */
    protected String getFirstSectionName()
    {
        return "section";
    }
    
    /**
     * Get the title of the first section of minisite
     * @return the section's title
     */
    protected I18nizableText getFirstSectionTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_MINISITE_FIRST_SECTION_TITLE");
    }
    
    @Override
    protected String getModulePageTemplate()
    {
        // the module page is a NODE page
        return null;
    }
    
    @Override
    protected void initializeModulePage(ModifiablePage modulePage)
    {
        modulePage.untag("SECTION");
        
        ModifiablePage firstSection = _createAndInitializeFirstSectionPage(modulePage, getFirstSectionName(), getFirstSectionTitle());
        if (firstSection != null)
        {
            _initializeFirstSectionDefaultZone(firstSection);
        }
    }
    
    /**
     * Create the first section of minisite if does not exist
     * @param modulePage the module page
     * @param name the name of section to create
     * @param pageTitle the page title
     * @return the created page or null if already exist
     */
    protected ModifiablePage _createAndInitializeFirstSectionPage(ModifiablePage modulePage, String name, I18nizableText pageTitle)
    {
        if (!modulePage.hasChild(name))
        {
            ModifiablePage page = modulePage.createChild(name, "ametys:defaultPage");
            
            // Title should not be missing, but just in case if the i18n message or the whole catalog does not exists in the requested language
            // to prevent a non-user-friendly error and still generate the project workspace.
            page.setTitle(StringUtils.defaultIfEmpty(_i18nUtils.translate(pageTitle, modulePage.getSitemapName()), "Missing title"));
            page.setType(PageType.CONTAINER);
            page.setSiteName(modulePage.getSiteName());
            page.setSitemapName(modulePage.getSitemapName());
            page.setTemplate(ProjectConstants.MINISITE_TEMPLATE);
            page.tag("SECTION");
            
            page.saveChanges();
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_PAGE_ADDED, _currentUserProvider.getUser(), eventParams));
            
            return page;
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Initialize the default zone for the first section of mini site
     * @param page The first page of minisite
     */
    protected void _initializeFirstSectionDefaultZone(ModifiablePage page)
    {
        ModifiableZone defaultZone = page.createZone("default");
        
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.CONTENT);
        
        try
        {
            ModifiableWorkflowAwareContent content = _createDefaultContent(page.getSite(), page.getSitemapName(), page.getTitle());
            defaultZoneItem.setContent(content);
            content.saveChanges();
        }
        catch (WorkflowException e)
        {
            getLogger().error("Unable to initialize the first page for minisite module, the page will not be editable until the content is manually created in the BackOffice", e);
        }
    }
    
    /**
     * Create a new content for a minisite page of the minisite module
     * @param site The site
     * @param sitemapName the name of the sitemap
     * @param title The content title
     * @return The content
     * @throws WorkflowException if an error occurred
     */
    protected ModifiableWorkflowAwareContent _createDefaultContent(Site site, String sitemapName, String title) throws WorkflowException
    {
        Map<String, Object> inputs = new HashMap<>();
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_TITLE_KEY, title);
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_NAME_KEY, NameHelper.filterName(title));
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_TYPES_KEY, new String[] {WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE});
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_LANGUAGE_KEY, sitemapName);
        inputs.put(CreateContentFunction.SITE_KEY, site.getName());
        
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow();
        workflow.initialize(WorkspacesConstants.CONTENT_WORKFLOW_NAME, __INITIAL_WORKFLOW_ACTION_ID, inputs);
    
        @SuppressWarnings("unchecked")
        Map<String, Object> results = (Map<String, Object>) inputs.get(AbstractWorkflowComponent.RESULT_MAP_KEY);
        ModifiableWorkflowAwareContent content = (ModifiableWorkflowAwareContent) results.get(Content.class.getName());
        
        return content;
    }
    
    @Override
    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of(ObservationConstants.EVENT_MINISITE_PAGE_CREATED,
                               ObservationConstants.EVENT_MINISITE_PAGE_UPDATED,
                               ObservationConstants.EVENT_MINISITE_PAGE_RENAMED,
                               ObservationConstants.EVENT_MINISITE_PAGE_DELETED);
    }

    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    { 
        if (isActive)
        {
            Map<String, Object> statistics = new HashMap<>();
            try
            {
                AmetysObjectIterable<Content> results = _contentSearcherFactory.create(WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE)
                        .search(new SiteQuery(project.getName()));
                statistics.put(__ARTICLE_NUMBER_HEADER_ID, results.getSize());
            }
            catch (Exception e)
            {
                getLogger().error("Error searching wall content in project " + project.getId(), e);
            }
            return statistics;
        }
        else
        {
            return Map.of(__ARTICLE_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }
    
    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__ARTICLE_NUMBER_HEADER_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_MINISITE_ARTICLE_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }
    
    @Override
    protected long _getModuleSize(Project project)
    {
        try
        {
            AmetysObjectIterable<Content> resultsIterable = _contentSearcherFactory.create(WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE)
                    .search(new SiteQuery(project.getName()));

            return resultsIterable.stream()
                .map(content -> content.getValue("illustration/image"))
                .filter(Objects::nonNull)
                .filter(Binary.class::isInstance)
                .map(Binary.class::cast)
                .mapToLong(Binary::getLength)
                .sum();
        }
        catch (Exception e)
        {
            getLogger().error("Error searching minisite content images in project " + project.getId(), e);
            return __SIZE_ERROR;
        }
    }

    @Override
    protected boolean _showModuleSize()
    {
        return true;
    }
}
