/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.group.GroupHelper;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Generator for the welcome mail of a new user
 */
public class MemberEventMailNotifierGenerator extends AbstractMailNotifierGenerator
{
    
    private GroupHelper _groupHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _groupHelper = (GroupHelper) smanager.lookup(GroupHelper.ROLE);
    }
    
    // This generator is called with an event as param instead of an activity
    // so we need to rework the generation
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        @SuppressWarnings("unchecked")
        Map<String, Object> parentContextAttr = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        Event event = (Event) parentContextAttr.get("event");
        Project project = (Project) parentContextAttr.get("project");
        XMLUtils.startElement(contentHandler, AbstractMailNotifierGenerator.ROOT_ELEMENT);
        
        XMLUtils.createElement(contentHandler, "eventKey", getEventIdAsI18nKey(event.getId()));
        
        saxProject(project);
        
        _saxMember(event, project);
        
        _saxProjectUserIdentity(event.getIssuer(), project, "issuer");
        
        XMLUtils.endElement(contentHandler, AbstractMailNotifierGenerator.ROOT_ELEMENT);
        contentHandler.endDocument();
    }
    
    private void _saxMember(Event event, Project project) throws SAXException
    {
        Map<String, Object> args = event.getArguments();
        
        MemberType type = (MemberType) args.get(ObservationConstants.ARGS_MEMBER_IDENTITY_TYPE);
        String identity = (String) args.get(ObservationConstants.ARGS_MEMBER_IDENTITY);
        
        if (MemberType.USER == type)
        {
            UserIdentity user = UserIdentity.stringToUserIdentity(identity);
            _saxProjectUserIdentity(user, project, "member");
        }
        else
        {
            // Group
            GroupIdentity group = GroupIdentity.stringToGroupIdentity(identity);
            _saxGroupIdentity(group, "member");
        }
    }
    
    private void _saxGroupIdentity(GroupIdentity group, String tagname) throws SAXException
    {
        XMLUtils.startElement(contentHandler, tagname);
        _groupHelper.saxGroupIdentity(group, contentHandler);
        XMLUtils.endElement(contentHandler, tagname);
    }
    
    @Override
    protected String _getModuleId()
    {
        // not used
        return null;
    }



}
