/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.modules;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;
import org.ametys.runtime.plugin.component.LazyInitializeExtensions;

/**
 * Extension point for Workspace Modules Managers
 */
@LazyInitializeExtensions
public class WorkspaceModuleExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<WorkspaceModule>
{
    /** Avalon Role */
    public static final String ROLE = WorkspaceModuleExtensionPoint.class.getName();
    
    private Set<String> _sortedIds;

    /**
     * Get a workspace module
     * @param moduleId The id of the module
     * @param <M> The module type
     * @return The workspace module
     */
    @SuppressWarnings("unchecked")
    public <M extends WorkspaceModule> M getModule(String moduleId)
    {
        return (M) getExtension(moduleId);
    }
    
    /**
     * Get the list of available modules
     * @return The modules
     */
    public List<WorkspaceModule> getModules()
    {
        return getExtensionsIds().stream().map(id -> getExtension(id)).collect(Collectors.toList());
    }
    
    /**
     * Get a workspace module by its name
     * @param moduleName The module name
     * @return The workspace module or null if not found
     */
    public WorkspaceModule getModuleByName(String moduleName)
    {
        List<WorkspaceModule> modules = getModules();
        for (WorkspaceModule workspaceModule : modules)
        {
            if (workspaceModule.getModuleName().equals(moduleName))
            {
                return workspaceModule;
            }
        }
        
        return null;
    }
    
    @Override
    public void initializeExtensions() throws Exception
    {
        super.initializeExtensions();
        
        Set<WorkspaceModule> sortedModules = new TreeSet<>();
        sortedModules.addAll(getModules());
        _sortedIds = sortedModules.stream().map(WorkspaceModule::getId).collect(Collectors.toCollection(LinkedHashSet::new));
    }
    
    @Override
    public Set<String> getExtensionsIds()
    {
        if (_sortedIds != null)
        {
            return _sortedIds;
        }
        else
        {
            return super.getExtensionsIds();
        }
    }
}
