/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.objects;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.apache.commons.lang3.ArrayUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.indexing.solr.SolrAclCacheUninfluentialObject;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.activities.ActivityHolder;
import org.ametys.plugins.repository.activities.ActivityHolderAmetysObject;
import org.ametys.plugins.repository.activities.DefaultActivityHolder;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObjectHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * {@link AmetysObject} for storing project informations.
 */
@SolrAclCacheUninfluentialObject
public class Project extends DefaultTraversableAmetysObject<ProjectFactory> implements ModifiableModelLessDataAwareAmetysObject, ActivityHolderAmetysObject
{
    /** Project node type name. */
    public static final String NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":project";
    
    /** Attribute name for project 's site */
    public static final String DATA_SITE = "site";

    private static final String __EXPLORER_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":resources";
    private static final String __DATA_TITLE = "title";
    private static final String __DATA_DESCRIPTION = "description";
    
    private static final String __DATA_MAILING_LIST = "mailingList";
    private static final String __DATA_CREATION = "creationDate";
    private static final String __DATA_MANAGERS = "managers";
    private static final String __DATA_MODULES = "modules";
    private static final String __DATA_INSCRIPTION_STATUS = "inscriptionStatus";
    private static final String __DATA_DEFAULT_PROFILE = "defaultProfile";
    private static final String __DATA_COVERIMAGE = "coverImage";
    private static final String __DATA_KEYWORDS = "keywords";
    private static final String __PLUGINS_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":plugins";
    private static final String __DATA_CATEGORIES = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":categories";

    /**
     * The inscription status of the project
     */
    public enum InscriptionStatus
    {
        /** Inscriptions are opened to anyone */
        OPEN("open"),
        /** Inscriptions are moderated */
        MODERATED("moderated"),
        /** Inscriptions are private */
        PRIVATE("private");
        
        private String _value;
        
        private InscriptionStatus(String value)
        {
            this._value = value;
        }  
           
        @Override
        public String toString() 
        {
            return _value;
        }
        
        /**
         * Converts a string to an Inscription
         * @param status The status to convert
         * @return The status corresponding to the string or null if unknown
         */
        public static InscriptionStatus createsFromString(String status)
        {
            for (InscriptionStatus v : InscriptionStatus.values())
            {
                if (v.toString().equals(status))
                {
                    return v;
                }
            }
            return null;
        }
    }
    
    /**
     * Creates a {@link Project}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link ProjectFactory} which creates the AmetysObject.
     */
    public Project(Node node, String parentPath, ProjectFactory factory)
    {
        super(node, parentPath, factory);
    }

    public ModifiableModelLessDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getProjectDataTypeExtensionPoint(), repositoryData);
    }
    
    public ActivityHolder getActivityHolder() throws RepositoryException
    {
        // The child node is provided by the activity-holder nodetype so we are sure it is there
        return new DefaultActivityHolder(getChild(ACTIVITIES_ROOT_NODE_NAME), _getFactory());
    }


    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        return getValue(__DATA_TITLE);
    }
    
    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        setValue(__DATA_TITLE, title);
    }
    
    /**
     * Retrieves the description.
     * @return the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getDescription() throws AmetysRepositoryException
    {
        return getValue(__DATA_DESCRIPTION);
    }
    
    /**
     * Set the description.
     * @param description the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setDescription(String description) throws AmetysRepositoryException
    {
        setValue(__DATA_DESCRIPTION, description);
    }
    
    /**
     * Remove the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void removeDescription() throws AmetysRepositoryException
    {
        removeValue(__DATA_DESCRIPTION);
    }
    
    /**
     * Retrieves the explorer nodes.
     * @return the explorer nodes or an empty {@link AmetysObjectIterable}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public ExplorerNode getExplorerRootNode() throws AmetysRepositoryException
    {
        return (ExplorerNode) getChild(__EXPLORER_NODE_NAME);
    }
    
    /**
     * Retrieves the explorer nodes.
     * @return the explorer nodes or an empty {@link AmetysObjectIterable}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public AmetysObjectIterable<ExplorerNode> getExplorerNodes() throws AmetysRepositoryException
    {
        return ((TraversableAmetysObject) getChild(__EXPLORER_NODE_NAME)).getChildren();
    }
    
    
    /**
     * Retrieves the mailing list.
     * @return the mailing list.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getMailingList() throws AmetysRepositoryException
    {
        return getValue(__DATA_MAILING_LIST);
    }
    
    /**
     * Set the mailing list.
     * @param mailingList the mailing list.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setMailingList(String mailingList) throws AmetysRepositoryException
    {
        setValue(__DATA_MAILING_LIST, mailingList);
    }
    
    /**
     * Remove the mailing list.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void removeMailingList() throws AmetysRepositoryException
    {
        removeValue(__DATA_MAILING_LIST);
    }
    
    /**
     * Retrieves the date of creation.
     * @return the date of creation.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public ZonedDateTime getCreationDate() throws AmetysRepositoryException
    {
        return getValue(__DATA_CREATION);
    }
    
    /**
     * Set the date of creation.
     * @param creationDate the date of creation
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setCreationDate(ZonedDateTime creationDate) throws AmetysRepositoryException
    {
        setValue(__DATA_CREATION, creationDate);
    }

    /**
     * Get the site of the project
     * @return The site. Can be null if the reference is broken.
     */
    public Site getSite()
    {
        try
        {
            SiteManager siteManager = _getFactory()._getSiteManager();
            Property weakReference = getNode().getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + DATA_SITE);
            String siteName = weakReference.getNode().getName();
            return siteManager.getSite(siteName);
        }
        catch (PathNotFoundException e)
        {
            _getFactory().getFactoryLogger().debug("Can not found site attribute for project " + getName(), e);
            return null;
        }
        catch (ItemNotFoundException e)
        {
            _getFactory().getFactoryLogger().debug("Can not found site reference for project " + getName(), e);
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unexpected repository exception", e);
        }
    }

    /**
     * Get the site of the project
     * @param site the site to set
     */
    public void setSite(Site site)
    {
        try
        {
            Node projectNode = getNode();
            if (projectNode.hasProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + DATA_SITE))
            {
                removeValue(DATA_SITE);
            }
            ValueFactory valueFactory = projectNode.getSession().getValueFactory();
            Value weakRefValue = valueFactory.createValue(site.getNode(), true);
            projectNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + DATA_SITE, weakRefValue);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unexpected repository exception", e);
        }
        
        if (needsSave())
        {
            saveChanges();
        }
    }
        
    /**
     * Get the project managers user identities
     * @return The managers
     */
    public UserIdentity[] getManagers()
    {
        return getValue(__DATA_MANAGERS, new UserIdentity[0]);
    }
    
    /**
     * Set the project managers
     * @param user The managers
     */
    public void setManagers(UserIdentity[] user)
    {
        setValue(__DATA_MANAGERS, user);
    }
    
    /**
     * Retrieve the list of activated modules for the project
     * @return The list of modules ids
     */
    public String[] getModules()
    {
        return getValue(__DATA_MODULES, new String[0]);
    }
    
    /**
     * Set the list of activated modules for the project
     * @param modules The list of modules
     */
    public void setModules(String[] modules)
    {
        setValue(__DATA_MODULES, modules);
    }
    
    /**
     * Add a module to the list of activated modules
     * @param moduleId The module id
     */
    public void addModule(String moduleId)
    {
        String[] modules = getValue(__DATA_MODULES);
        if (!ArrayUtils.contains(modules, moduleId))
        {
            setValue(__DATA_MODULES, modules == null ? new String[]{moduleId} : ArrayUtils.add(modules, moduleId));
        }
    }
    
    /**
     * Remove a module from the list of activated modules
     * @param moduleId The module id
     */
    public void removeModule(String moduleId)
    {
        String[] modules = getValue(__DATA_MODULES);
        if (ArrayUtils.contains(modules, moduleId))
        {
            setValue(__DATA_MODULES, ArrayUtils.removeElement(modules, moduleId));
        }
    }
    
    /**
     * Get the inscription status of the project
     * @return The inscription status
     */
    public InscriptionStatus getInscriptionStatus()
    {
        if (hasValue(__DATA_INSCRIPTION_STATUS))
        {
            return InscriptionStatus.createsFromString(getValue(__DATA_INSCRIPTION_STATUS));
        }
        return InscriptionStatus.PRIVATE;
    }
    
    /**
     * Set the inscription status of the project
     * @param inscriptionStatus The inscription status
     */
    public void setInscriptionStatus(String inscriptionStatus)
    {
        if (inscriptionStatus != null && InscriptionStatus.createsFromString(inscriptionStatus) != null)
        {
            setValue(__DATA_INSCRIPTION_STATUS, inscriptionStatus);
        }
        else
        {
            removeValue(__DATA_INSCRIPTION_STATUS);
        }
    }

    /**
     * Set the inscription status of the project
     * @param inscriptionStatus The inscription status
     */
    public void setInscriptionStatus(InscriptionStatus inscriptionStatus)
    {
        if (inscriptionStatus != null)
        {
            setValue(__DATA_INSCRIPTION_STATUS, inscriptionStatus.toString());
        }
        else
        {
            removeValue(__DATA_INSCRIPTION_STATUS);
        }
    }

    /**
     * Get the default profile for new members of the project
     * @return The default profile
     */
    public String getDefaultProfile()
    {
        return getValue(__DATA_DEFAULT_PROFILE);
    }
     
    
    /**
     * Set the default profile for the members of the project
     * @param profileId The ID of the profile
     */
    public void setDefaultProfile(String profileId)
    {
        if (profileId != null)
        {
            setValue(__DATA_DEFAULT_PROFILE, profileId);
        }
        else
        {
            removeValue(__DATA_DEFAULT_PROFILE);
        }
    }
    
    /**
     * Get the root for plugins
     * @return the root for plugins
     * @throws AmetysRepositoryException if an error occurs.
     */
    public ModifiableTraversableAmetysObject getRootPlugins () throws AmetysRepositoryException
    {
        return (ModifiableTraversableAmetysObject) getChild(__PLUGINS_NODE_NAME);
    }

    /**
     * Retrieve the list of tags 
     * @return The list of tags
     * @throws AmetysRepositoryException if an error occurs
     */
    public Set<String> getTags() throws AmetysRepositoryException
    {
        return TaggableAmetysObjectHelper.getTags(this);
    }
    
    /**
     * Add a tag to the project
     * @param tag The tag
     * @throws AmetysRepositoryException if an error occurs
     */
    public void tag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.tag(this, tag);
    }

    /**
     * Remove a tag from the project
     * @param tag The tag
     * @throws AmetysRepositoryException if an error occurs
     */
    public void untag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.untag(this, tag);
    }

    /**
     * Set the project tags
     * @param tags The list of tags
     */
    public void setTags(List<String> tags)
    {
        Set<String> currentTags = getTags();
        // remove old tags not selected
        currentTags.stream()
                .filter(tag -> !tags.contains(tag))
                .forEach(tag -> untag(tag));
        
        // add new selected tags
        tags.stream()
                .filter(tag -> !currentTags.contains(tag))
                .forEach(tag -> tag(tag));
    }


    /**
     * Retrieve the list of categories
     * @return The categories
     * @throws AmetysRepositoryException if an error occurs
     */
    public Set<String> getCategories() throws AmetysRepositoryException
    {
        return TaggableAmetysObjectHelper.getTags(this, __DATA_CATEGORIES);
    }
    
    /**
     * Add a category to the project
     * @param category The category
     * @throws AmetysRepositoryException if an error occurs
     */
    public void addCategory(String category) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.tag(this, category, __DATA_CATEGORIES);
    }

    /**
     * Remove a category from the project
     * @param category The category
     * @throws AmetysRepositoryException if an error occurs
     */
    public void removeCategory(String category) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.untag(this, category, __DATA_CATEGORIES);
    }
    
    /**
     * Set the category tags of the project
     * @param categoryTags The category tags
     */
    public void setCategoryTags(List<String> categoryTags)
    {
        Set<String> currentCategories = getCategories();
        // remove old tags not selected
        currentCategories.stream()
                .filter(category -> !categoryTags.contains(category))
                .forEach(category -> removeCategory(category));
        
        // add new selected tags
        categoryTags.stream()
                .filter(category -> !currentCategories.contains(category))
                .forEach(category -> addCategory(category));
    }
    
    /**
     * Set the cover image of the site
     * @param is The input stream of the cover image
     * @param mimeType The mimetype of the cover image
     * @param filename The filename of the cover image
     * @param lastModificationDate The last modification date of the cover image
     * @throws IOException if an error occurs while setting the cover image
     */
    public void setCoverImage(InputStream is, String mimeType, String filename, ZonedDateTime lastModificationDate) throws IOException
    {
        if (is != null)
        {
            Binary coverImage = new Binary();
            coverImage.setInputStream(is);
            Optional.ofNullable(mimeType).ifPresent(coverImage::setMimeType);
            Optional.ofNullable(filename).ifPresent(coverImage::setFilename);
            Optional.ofNullable(lastModificationDate).ifPresent(coverImage::setLastModificationDate);
            setValue(__DATA_COVERIMAGE, coverImage);
        }
        else
        {
            removeValue(__DATA_COVERIMAGE);
        }
    }
    
    /**
     * Returns the cover image of the project
     * @return the cover image of the project
     */
    public Binary getCoverImage()
    {
        return getValue(__DATA_COVERIMAGE);
    }
    
    /**
     * Generates SAX events for the project's cover image
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @throws SAXException if error occurs during the SAX events generation
     * @throws IOException if an I/O error occurs while reading the cover image
     */
    public void coverImageToSAX(ContentHandler contentHandler) throws SAXException, IOException
    {
        dataToSAX(contentHandler, __DATA_COVERIMAGE);
    }
    
    /**
     * Retrieve the list of keywords for the project
     * @return The list of keywords
     */
    public String[] getKeywords()
    {
        return getValue(__DATA_KEYWORDS, new String[0]);
    }
    
    /**
     * Set the list of keywordss for the project
     * @param keywords The list of keywords
     */
    public void setKeywords(String[] keywords)
    {
        setValue(__DATA_KEYWORDS, keywords);
    }
}
