/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.observers;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract {@link Observer} to update ACL solr cache for contents
 */
public abstract class AbstractUpdateAclSorCacheObserver  extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The maetys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The solr indexer */
    protected SolrIndexer _solrIndexer;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _solrIndexer = (SolrIndexer) smanager.lookup(SolrIndexer.ROLE);
    }

    public int getPriority()
    {
        return MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        String projectId = (String) event.getArguments().get(ObservationConstants.ARGS_PROJECT_ID);
        Project project = _resolver.resolveById(projectId);
        
        Expression contentsExpr = getContentsExpression(event, project);
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(contentsExpr);
        AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);
        _solrIndexer.updateAclCache(contents);
    }
    
    /**
     * Get the JCR expression corresponding to the contents that must be updated
     * @param event the event
     * @param project the project
     * @return the the JCR expression
     */
    protected abstract Expression getContentsExpression(Event event, Project project);
    
}
