/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.convertor;

import java.util.Collections;
import java.util.List;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightContextConvertor;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.web.repository.content.WebContent;

/**
 * This {@link RightContextConvertor} delegates rights to workspace module root
 */
public class ContentToModuleRightContextConvertor implements RightContextConvertor, Serviceable
{
    /** ContentTypes Helper */
    protected ContentTypesHelper _cTypeHelper;
    /** Project Manager */
    protected ProjectManager _projectManager;
    /** Project right helper */
    protected ProjectRightHelper _projectRightHelper;
    
    private ServiceManager _manager;
    

    
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
    }
    
    /**
     * Get the project right helper instance
     * @return the project right helper instance
     */
    protected synchronized ProjectRightHelper getProjectRightHelper()
    {
        if (_projectRightHelper == null)
        {
            try
            {
                _projectRightHelper = (ProjectRightHelper) _manager.lookup(ProjectRightHelper.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _projectRightHelper;
        
    }
    
    /**
     * Get the project manager instance
     * @return the project manager instance
     */
    protected synchronized ProjectManager getProjectManager()
    {
        if (_projectManager == null)
        {
            try
            {
                _projectManager = (ProjectManager) _manager.lookup(ProjectManager.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _projectManager;
        
    }
    
    /**
     * Get the content types helper instance
     * @return the content types helper instance
     */
    protected synchronized ContentTypesHelper getContentTypesHelper() 
    {
        if (_cTypeHelper == null)
        {
            try
            {
                _cTypeHelper = (ContentTypesHelper) _manager.lookup(ContentTypesHelper.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _cTypeHelper;
    }
    
    @SuppressWarnings("unchecked")
    public Set<Object> convert(Object object)
    {
        Set<? extends Object> result = _toModuleRoots(object);
        return (Set<Object>) result;
    }
    
    /**
     * Get the module roots associated to current object
     * @param object The object context
     * @return the module roots associated to current object or empty set if the object is not linked to a existing module
     */
    protected Set<ResourceCollection> _toModuleRoots(Object object)
    {
        if (object instanceof WebContent)
        {
            WorkspaceModule module = _getWorkspaceModule((Content) object);
            if (module != null)
            {
                List<Project> projects = getProjectManager().getProjectsForSite(((WebContent) object).getSite());
                return projects.stream()
                    .map(p -> module.getModuleRoot(p, false))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());
            }
        }
        
        return Collections.EMPTY_SET;
    }
    
    /**
     * Get the workspace module associated to a content
     * @param content the content
     * @return the workspace module or null if the content does not match any module
     */
    protected WorkspaceModule _getWorkspaceModule(Content content)
    {
        for (Entry<String, WorkspaceModule> entry : getProjectRightHelper().getProjectContentTypesAndModules().entrySet())
        {
            if (getContentTypesHelper().isInstanceOf(content, entry.getKey()))
            {
                return entry.getValue();
            }
        }
        
        return null;
    }

}
