/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.query.expression.Expression.LogicalOperator;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Abstract generator for search module in the repository
 *
 */
public abstract class AbstractXpathSearchModuleGenerator extends AbstractSearchModuleGenerator
{

    @Override
    protected void saxHits(String siteName, String lang, String textfield, Request request, int offset, int limit, int minLimit) throws SAXException, ProcessingException
    {
        try
        {
            String query = getXPathQuery(siteName, lang, textfield, request, offset, limit);
            if (StringUtils.isNotEmpty(query))
            {
                AmetysObjectIterable<AmetysObject> searchResults = _resolver.query(query);
                long totalCount = searchResults.getSize();
                
                // I don't like this, but I did not found any better way
                List<AmetysObject> results = searchResults.stream()
                    .skip(offset)
                    .limit(limit)
                    .collect(Collectors.toList());
                
                saxHits(results, lang, offset, limit, minLimit, totalCount);
            }
            else
            {
                saxHits(Collections.EMPTY_LIST, lang, offset, limit, minLimit, 0);
            }
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to get search results", e);
        }
    }
    
    /**
     * Get the xpath query on project
     * @param projects the targeted projects
     * @return the xpath query
     */
    protected String getProjectXPathQuery(List<Project> projects)
    {
        List<String> projectNames = projects.stream()
                .map(Project::getName)
                .map(name -> "fn:name() = '" + name + "'")
                .collect(Collectors.toList());
        
        return "//element(*, ametys:project)" + (!projectNames.isEmpty() ? "[" + String.join(LogicalOperator.OR.toString(), projectNames) + "]" : "");
    }
    
    /**
     * Get the Xpath query
     * @param siteName the current site name
     * @param lang the current language
     * @param textfield the search input
     * @param request the request
     * @param offset the search offset
     * @param limit the max number of results
     * @return the search results
     * @throws Exception if the search failed
     */
    protected abstract String getXPathQuery(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception;
    
}
