/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SortOrder;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ContentLanguageQuery;
import org.ametys.cms.search.query.FullTextQuery;
import org.ametys.cms.search.query.JoinQuery;
import org.ametys.cms.search.query.MatchAllQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.plugins.userdirectory.transformation.xslt.UserXSLTHelper;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.indexing.solr.SolrWorkspacesConstants;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Generator for members search module
 *
 */
public class MemberSearchModuleGenerator extends AbstractContentSolrSearchModuleGenerator
{
    @Override
    protected SimpleContentSearcher getSearcher()
    {
        return _contentSearcherFactory.create(WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
    }
    
    @Override
    protected String getSortFieldName()
    {
        return "lastname";
    }
    
    @Override
    protected SortOrder getSortOrder()
    {
        return SortOrder.ASC;
    }
    
    @Override
    protected Query getQuery(String siteName, String lang, String textfield, Request request)
    {
        Collection<Query> queries = new HashSet<>();
        
        queries.add(getProjectQuery(request));
        queries.add(new ContentLanguageQuery(lang));
        
        if (StringUtils.isNotBlank(textfield))
        {
            Query titleQuery = new StringQuery(SolrFieldNames.TITLE, Operator.SEARCH, textfield.trim(), lang);
            Query fullTextQuery = new FullTextQuery(textfield.trim(), SolrFieldNames.FULL, lang, Operator.SEARCH);
            
            Query skillsQuery = new JoinQuery(fullTextQuery, "skills");
            Query keywordsQuery = new JoinQuery(fullTextQuery, "keywords");
            
            Query orQuery = new OrQuery(titleQuery, fullTextQuery, skillsQuery, keywordsQuery);
            queries.add(orQuery);
        }
        
        Query fullQuery;
        
        if (!queries.isEmpty())
        {
            fullQuery = new AndQuery(queries);
        }
        else
        {
            fullQuery = new MatchAllQuery();
        }
        
        return fullQuery;
    }
    
    @Override
    protected void saxPage(Content content) throws SAXException
    {
        String userPageId = UserXSLTHelper.getUserPage(content.getId(), _projectManager.getCatalogSiteName());
        if (userPageId != null)
        {
            XMLUtils.createElement(contentHandler, "page", userPageId);
        }
    }
    
    /**
     * Returns the project query
     * @param request the request
     * @return the project query
     */
    protected Query getProjectQuery(Request request)
    {
        List<Project> projects = getProjects(request, false);
        
        Query projectQuery = projects.stream()
                .map(Project::getId)
                .map(id -> new StringQuery(SolrWorkspacesConstants.PROJECT_ID, Operator.EQ , id, null))
                .collect(OrQuery.collector());
            
        return projectQuery;
    }
}
