/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections4.MapUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.jcr.JCRTasksList;
import org.ametys.plugins.workspaces.tasks.jcr.JCRTasksListFactory;
import org.ametys.plugins.workspaces.tasks.json.TasksListJSONHelper;

/**
 * DAO for interacting with tasks list of a project
 */
public class WorkspaceTasksListDAO extends AbstractWorkspaceTaskDAO
{
    /** The Avalon role */
    public static final String ROLE = WorkspaceTasksListDAO.class.getName();
    
    /** The tasks list JSON helper */
    protected TasksListJSONHelper _tasksListJSONHelper;
    
    /** The tasks list colors component */ 
    protected TasksListColorsComponent _tasksListColorsComponent;
    
    /** The tasks list icons component */
    protected TasksListIconsComponent _tasksListIconsComponent;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _tasksListJSONHelper = (TasksListJSONHelper) manager.lookup(TasksListJSONHelper.ROLE);
        _tasksListColorsComponent = (TasksListColorsComponent) manager.lookup(TasksListColorsComponent.ROLE);
        _tasksListIconsComponent = (TasksListIconsComponent) manager.lookup(TasksListIconsComponent.ROLE);
    }

    /**
     * Get the tasks lists from the project
     * @return the tasks lists
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getTasksLists()
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        
        _checkReadAccess(project, TasksWorkspaceModule.TASK_MODULE_ID);
        
        List<Map<String, Object>> tasksListsInfo = new ArrayList<>();
        for (TasksList tasksList : _getProjectTasksLists(project))
        {
            tasksListsInfo.add(_tasksListJSONHelper.tasksListAsJson(tasksList));
        }
        
        return tasksListsInfo;
    }

    /**
     * Add a new tasks list to the project
     * @param label the tasks list label
     * @return The tasks list data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addTasksList(String label)
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        
        ModifiableTraversableAmetysObject tasksRoot = _getTasksListsRoot(project, true);
        _checkUserRights(tasksRoot, RIGHTS_HANDLE_TASK_LIST);

        int index = 1;
        String name = "tasks-list-1";
        while (tasksRoot.hasChild(name))
        {
            index++;
            name = "tasks-list-" + index;
        }
        
        List<TasksList> projectsTasksList = _getProjectTasksLists(project);
        JCRTasksList tasksList = (JCRTasksList) tasksRoot.createChild(name, JCRTasksListFactory.TASKS_LIST_NODETYPE);
        
        tasksList.setListId(tasksList.getId());
        tasksList.setPosition(Long.valueOf(projectsTasksList.size()));
        tasksList.setLabel(label);
        tasksList.setColor(_tasksListColorsComponent.getDefaultKey());
        tasksList.setIcon(_tasksListIconsComponent.getDefaultKey());
        
        ZonedDateTime now = ZonedDateTime.now();
        tasksList.setCreationDate(now);
        tasksList.setLastModified(now);
        tasksList.setAuthor(_currentUserProvider.getUser());
        
        tasksRoot.saveChanges();
        
        return _tasksListJSONHelper.tasksListAsJson(tasksList);
    }
    
    /**
     * Edit a tasks list
     * @param tasksListId The id of the tasks list to edit
     * @param newLabel The tasks list new label
     * @return The tasks list data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editTasksList(String tasksListId, String newLabel)
    {
        JCRTasksList tasksList = _resolver.resolveById(tasksListId);

        // Check user right
        _checkUserRights(tasksList.getParent(), RIGHTS_HANDLE_TASK_LIST);
        
        Map<String, Object> results = new HashMap<>();
         
        if (!newLabel.equals(tasksList.getLabel()))
        {
            tasksList.setLabel(newLabel);
            tasksList.setLastModified(ZonedDateTime.now());
            tasksList.saveChanges();
        }
         
        results.put("id", tasksList.getId());
        return results;
    }
    
    /**
     * Delete a tasks list
     * @param tasksListId the tasks list id to remove
     * @return the tasks list data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteTasksList(String tasksListId)
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        
        TasksList taskList = _resolver.resolveById(tasksListId);

        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = taskList.getParent();
        _checkUserRights(tasksRoot, RIGHTS_DELETE_TASK_LIST);
        
        if (!getChildTask(tasksListId).isEmpty())
        {
            throw new UnsupportedOperationException("Can't remove tasks list with id '" + tasksListId + "' because it is not empty");
        }
        
        taskList.remove();
        
        // Reorder tasks lists position
        long position = 0;
        for (TasksList childTasksList : _getProjectTasksLists(project))
        {
            childTasksList.setPosition(position);
            position++;
        }
        
        tasksRoot.saveChanges();
        
        return MapUtils.EMPTY_SORTED_MAP;
    }
    
    /**
     * Move tasks list to new position
     * @param tasksListId the tasks list id to move
     * @param newPosition the new position
     * @return The tasks list data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> moveTasksList(String tasksListId, long newPosition)
    {
        Project project = _workspaceHelper.getProjectFromRequest();
        
        TasksList taskList = _resolver.resolveById(tasksListId);
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = taskList.getParent();
        _checkUserRights(tasksRoot, RIGHTS_HANDLE_TASK_LIST);
        
        Map<String, Object> results = new HashMap<>();

        List<TasksList> childTasksLists = _getProjectTasksLists(project);
        int size = childTasksLists.size();
        if (newPosition > size)
        {
            throw new IllegalArgumentException("New position (" + newPosition + ") can't be greater than tasks lists size (" + size + ")");
        }
        
        long position = 0;
        for (TasksList childTasksList : childTasksLists)
        {
            if (position == newPosition)
            {
                position++;
            }
            
            if (childTasksList.getId().equals(tasksListId))
            {
                childTasksList.setPosition(newPosition);
            }
            else
            {
                childTasksList.setPosition(position);
                position++;
            }
        }
        
        tasksRoot.saveChanges();
        
        return results;
    }
    
    /**
     * Get all available colors for the tasks list
     * @return the colors
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, String>> getTasksListColors()
    {
        return _tasksListColorsComponent.getColors()
                    .entrySet()
                    .stream()
                    .map(e -> Map.of("id", e.getKey(), "color", e.getValue().get("main")))
                    .collect(Collectors.toList());
    }
    
    /**
     * Get all available icons for the tasks list
     * @return the icons
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, String>> getTasksListIcons()
    {
        return _tasksListIconsComponent.getIcons();
    }
    
    /**
     * Update the tasks list color
     * @param tasksListId the tasks list id
     * @param colorId the color id
     * @return The tasks list data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> updateTasksListColor(String tasksListId, String colorId)
    {
        JCRTasksList tasksList = _resolver.resolveById(tasksListId);
        
        // Check user right
        _checkUserRights(tasksList.getParent(), RIGHTS_HANDLE_TASK_LIST);
        
        Map<String, Object> results = new HashMap<>();
         
        if (!colorId.equals(tasksList.getColor()))
        {
            tasksList.setColor(colorId);
            tasksList.setLastModified(ZonedDateTime.now());
            tasksList.saveChanges();
        }
         
        results.put("id", tasksList.getId());
        return results;
    }
    
    /**
     * Update the tasks list icon
     * @param tasksListId the tasks list id
     * @param iconId the icon id
     * @return The tasks list data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> updateTasksListIcon(String tasksListId, String iconId)
    {
        JCRTasksList tasksList = _resolver.resolveById(tasksListId);
        
        // Check user right
        _checkUserRights(tasksList.getParent(), RIGHTS_HANDLE_TASK_LIST);
        
        Map<String, Object> results = new HashMap<>();
         
        if (!iconId.equals(tasksList.getIcon()))
        {
            tasksList.setIcon(iconId);
            tasksList.setLastModified(ZonedDateTime.now());
            tasksList.saveChanges();
        }
         
        results.put("id", tasksList.getId());
        return results;
    }
    
    /**
     * Get the project tasks lists
     * @param project the project
     * @return the list of tasks lists
     */
    protected List<TasksList> _getProjectTasksLists(Project project)
    {
        TasksWorkspaceModule taskModule = _workspaceModuleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        return Optional.of(project)
            .map(p -> taskModule.getTasksListsRoot(p, true))
            .map(ModifiableTraversableAmetysObject::getChildren)
            .map(AmetysObjectIterable::stream)
            .orElseGet(Stream::empty)
            .filter(TasksList.class::isInstance)
            .map(TasksList.class::cast)
            .sorted(Comparator.comparingLong(TasksList::getPosition))
            .collect(Collectors.toList());
    }
    
    /**
     * Get the child task of the tasks list
     * @param tasksListId the tasks list id
     * @return the list of task
     */
    public List<Task> getChildTask(String tasksListId)
    {
        String xpathQuery = "//element(*, ametys:task)[ametys:tasksListId = '" + tasksListId + "']";
        return _resolver.query(xpathQuery)
            .stream()
            .filter(Task.class::isInstance)
            .map(Task.class::cast)
            .sorted(Comparator.comparingLong(Task::getPosition))
            .collect(Collectors.toList());
    }
}
