/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.jcr;

import javax.jcr.Node;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObjectFactory;
import org.ametys.plugins.workspaces.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.workspaces.tasks.WorkspaceTasksListDAO;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link AmetysObjectFactory} for handling {@link JCRTasksList}.
 */
public class JCRTasksListFactory extends DefaultTraversableAmetysObjectFactory
{
    /** JCR nodetype for tasks */
    public static final String TASKS_LIST_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":tasks-list";
    
    /** The task list model */
    private Model _taskListModel;

    /** The task list DAO */
    private WorkspaceTasksListDAO _workspaceTasksListDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workspaceTasksListDAO = (WorkspaceTasksListDAO) manager.lookup(WorkspaceTasksListDAO.ROLE);
    }

    @Override
    public JCRTasksList getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new JCRTasksList(node, parentPath, this);
    }
    
    /**
     * Get the task list model
     * @return the task list model
     */
    public Model getTaskListModel()
    {
        if (_taskListModel == null)
        {
            try
            {
                String role = ModelItemTypeExtensionPoint.ROLE_TASK_LIST;
                _taskListModel = Model.of(
                    "task.list.model.id", 
                    "task.list.model.family.id",
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_ID, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_POSITION, false, ModelItemTypeConstants.LONG_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_LABEL, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_COLOR_ID, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_ICON_ID, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_AUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_CREATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                    DefaultElementDefinition.of(JCRTasksList.ATTRIBUTE_LASTMODIFIED, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role)
                );
            }
            catch (Exception e) 
            {
                getLogger().error("An error occurred getting the task list model", e);
                throw new RuntimeException("An error occurred getting the task list model", e);
            }
        }
        return _taskListModel;
    }

    /**
     * Get the task model
     * @return the task model
     */
    public WorkspaceTasksListDAO getWorkspaceTasksListDAO()
    {
        return _workspaceTasksListDAO;
    }
}
