/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.actions;

import java.time.LocalDate;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.ObjectModelHelper;

import org.ametys.cms.tag.Tag;
import org.ametys.plugins.calendar.events.EventsFilter;
import org.ametys.plugins.calendar.events.EventsFilterHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.filter.WebContentFilter;
import org.ametys.web.filter.WebContentFilter.AccessLimitation;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;

/**
 * This action creates a filter for RSS calendar
 *
 */
public class SetFilterInRequestAttributesAction extends org.ametys.web.repository.page.actions.SetFilterInRequestAttributesAction implements Contextualizable
{
    /** The events helper. */
    protected EventsFilterHelper _eventsFilterHelper;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _ametysResolver;
    
    /** The avalon context */
    protected Context _context;

    private String _view;
    private String _rangeType;
    private Set<Tag> _categories;
    private int _year;
    private int _month;
    private int _day;
    private int _monthsBefore;
    private int _monthsAfter;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _eventsFilterHelper = (EventsFilterHelper) smanager.lookup(EventsFilterHelper.ROLE);
        _ametysResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    /**
     * Get the content filter from a ZoneItem.
     * 
     * @param parameters the action parameters, including the ZoneItem id.
     * @param result the action result.
     * @return the content filter.
     * @throws ParameterException if parameters are invalid
     */
    @Override
    protected WebContentFilter _getFilterFromZoneItem(Parameters parameters, Map<String, String> result) throws ParameterException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> parentContextAttrs = (Map<String, Object>) ContextHelper.getObjectModel(_context).get(ObjectModelHelper.PARENT_CONTEXT);
        if (parentContextAttrs == null)
        {
            parentContextAttrs = Collections.EMPTY_MAP;
        }
        
        WebContentFilter filter = null;

        String zoneItemId = parameters.getParameter("zoneItemId");
        _view = parameters.getParameter("view");

        LocalDate today = LocalDate.now();
        _year = parameters.getParameterAsInteger("year", today.getYear());
        _month = parameters.getParameterAsInteger("month", today.getMonthValue());
        _day = parameters.getParameterAsInteger("day", today.getDayOfMonth());

        ZoneItem zoneItem = _resolver.resolveById("zoneItem://" + zoneItemId);
        if (_isValid(zoneItem))
        {
            _monthsBefore = Math.toIntExact(zoneItem.getServiceParameters().getValue("months-before", true, 3L));
            _monthsAfter = Math.toIntExact(zoneItem.getServiceParameters().getValue("months-after", true, 3L));

            // Creates filter from services parameters
            _rangeType = parameters.getParameter("rangeType", _eventsFilterHelper.getDefaultRangeType(zoneItem));

            String siteName = zoneItem.getZone().getSitemapElement().getSiteName();
            
            // Get the search context to match, from the zone item or from the parameters.
            @SuppressWarnings("unchecked")
            List<Map<String, Object>> searchValues = _eventsFilterHelper.getSearchContext(zoneItem, (List<Map<String, Object>>) parentContextAttrs.get("search"));
            _categories = _eventsFilterHelper.getTagCategories(zoneItem, searchValues, siteName);
            
            filter = _getFilterFromZoneItem(zoneItem);
        }

        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        result.put("siteName", sitemapElement.getSiteName());
        result.put("lang", sitemapElement.getSitemapName());
        result.put("path", sitemapElement.getPathInSitemap());

        return filter;
    }

    @Override
    protected WebContentFilter _getFilterFromZoneItem(ZoneItem zoneItem) throws ParameterException
    {
        String zoneItemId = zoneItem.getId();
        
        EventsFilter filter = new EventsFilter(zoneItemId, _ametysResolver, _contentTypeEP, _siteManager, _tagProviderEP);
        filter.addSortCriteria(EventsFilterHelper.START_DATE_META, false, true);
        filter.addSortCriteria(EventsFilterHelper.END_DATE_META, false, true);
        filter.setLength(Integer.MAX_VALUE);
        String type = "agenda";

        EventsFilterHelper.DateTimeRange dateRange = _eventsFilterHelper.getDateRange(type, _year, _month, _day, _monthsBefore, _monthsAfter, _rangeType);
        Expression expression = _eventsFilterHelper.getExpression(type, dateRange);

        Set<String> contentTypes = _eventsFilterHelper.getContentTypes(zoneItem);

        boolean maskOrphan = _eventsFilterHelper.getMaskOrphan(zoneItem);
        
        AccessLimitation accessLimitation = _eventsFilterHelper.getAccessLimitation(zoneItem);
        
        // Get all tags belonging to the categories.
        Set<String> categoryTags = _eventsFilterHelper.getAllTags(_categories);

        filter.setTitle(new I18nizableText(_eventsFilterHelper.getTitle(zoneItem)));
        
        List<Map<String, Object>> searchValues = _eventsFilterHelper.getSearchContext(zoneItem, null);
        
        // Configure the filter
        _eventsFilterHelper.configureFilter(filter, expression, contentTypes, searchValues, categoryTags, _view, maskOrphan, accessLimitation);

        return filter;
    }

    /**
     * Test if the ZoneItem is valid.
     * 
     * @param zoneItem the ZoneItem.
     * @return true if the ZoneItem is valid, false otherwise.
     */
    @Override
    protected boolean _isValid(ZoneItem zoneItem)
    {
        return zoneItem.getType().equals(ZoneItem.ZoneType.SERVICE) && "org.ametys.plugins.calendar.Agenda".equals(zoneItem.getServiceId());
    }
}
