/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.events;

import java.util.HashSet;
import java.util.Set;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.tags.TagExpression;

/**
 * Content filter which allows to directly provide:
 * - a custom metadata expression to filter the contents
 * - a list of tags to match of only one is required for the content to match ("OR" tags).
 */
public class EventsFilter extends org.ametys.web.filter.StaticWebContentFilter
{
    /** The metadata expression. */
    protected Expression _metadataExpression;
    
    /**
     * Build an events filter.
     */
    public EventsFilter()
    {
        super();
    }
    
    /**
     * Build an events filter.
     * @param id The filter id
     * @param resolver The ametys object resolver
     * @param contentTypeExtensionPoint The extension point for content types
     * @param siteManager The site manager
     * @param tagProviderEP The tag provider
     */
    public EventsFilter(String id, AmetysObjectResolver resolver, ContentTypeExtensionPoint contentTypeExtensionPoint, SiteManager siteManager, TagProviderExtensionPoint tagProviderEP)
    {
        super(id, resolver, contentTypeExtensionPoint, siteManager, tagProviderEP);
    }
    
    /**
     * Build a events filter from copy a another
     * @param id The filter id
     * @param originalFilter The filter to copy
     * @param resolver The ametys object resolver
     * @param contentTypeExtensionPoint The extension point for content types
     * @param siteManager The site manager
     * @param tagProviderEP The tag provider
     */
    public EventsFilter(String id, EventsFilter originalFilter, AmetysObjectResolver resolver, ContentTypeExtensionPoint contentTypeExtensionPoint, SiteManager siteManager, TagProviderExtensionPoint tagProviderEP)
    {
        super(id, originalFilter, resolver, contentTypeExtensionPoint, siteManager, tagProviderEP);
    }
    
    /**
     * Set the metadata expression.
     * @param expression the metadata expression.
     */
    public void setMetadataExpression(Expression expression)
    {
        _metadataExpression = expression;
    }
    
    @Override
    public Expression getMetadataExpression()
    {
        if (_metadataExpression == null)
        {
            return super.getMetadataExpression();
        }
        else
        {
            return _metadataExpression;
        }
    }
    
    @Override
    public FilterSearchContext createSeachContext()
    {
        return new EventFilterSearchContext(_siteManager);
    }
    
    /**
     * Clear the search contexts
     */
    public void clearSearchContexts()
    {
        _searchContexts.clear();
    }
    
    /**
     * FilterSearchContext specific to the events filter.
     */
    public class EventFilterSearchContext extends DefaultFilterSearchContext
    {
        /** The tags. */
        protected Set<String> _orTags;
        
        /**
         * Build an EventFilterSearchContext.
         * @param siteManager the site manager.
         */
        protected EventFilterSearchContext(SiteManager siteManager)
        {
            super(siteManager);
            _orTags = new HashSet<>();
        }
        
        /**
         * Get the OR tags.
         * @return the OR tags.
         */
        public Set<String> getOrTags()
        {
            return _orTags;
        }
        
        /**
         * Set the OR tags.
         * @param tags the OR tags to set.
         */
        public void setOrTags(Set<String> tags)
        {
            _orTags = new HashSet<>(tags);
        }
        
        /**
         * Add a tag to the OR list.
         * @param tagName the name of the tag to add.
         */
        public void addOrTag(String tagName)
        {
            _orTags.add(tagName);
        }
        
        /**
         * Get the expression corresponding to the filter's tags
         * @param siteName The current site name
         * @return The expression corresponding to the filter's tags
         */
        @Override
        public Expression getTagsExpression(String siteName)
        {
            Expression tagExpr = super.getTagsExpression(siteName);
            Expression expr = null;
            
            if (!_orTags.isEmpty())
            {
                Expression[] tagExprArray = new Expression[_orTags.size()];
                int i = 0;
                for (String tag : _orTags)
                {
                    tagExprArray[i] = new TagExpression(Operator.EQ, tag);
                    i++;
                }
                expr = tagExpr != null ? new AndExpression(tagExpr, new OrExpression(tagExprArray)) : new OrExpression(tagExprArray);
            }
            else
            {
                expr = tagExpr;
            }
            
            return expr;
        }
    }
    
    
}
