<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2022 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                xmlns:dateutils="org.ametys.core.util.DateUtils"
                xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
                xmlns:escaper="org.apache.commons.text.StringEscapeUtils"
                xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
                xmlns:exsl="http://exslt.org/common"
                xmlns:encoder="java.net.URLEncoder"
                extension-element-prefixes="resolver escaper ametys encoder exsl dateutils">
                
    <xsl:import href="service:calendar://pages/services/search/common/common.xsl"/>
    
    <!-- the max number of events visible within a day -->
    <xsl:variable name="dayMaxEvents">3</xsl:variable>
    
    <!-- max resolution to switch to mobile view -->
    <xsl:variable name="mobileMaxResolution">768</xsl:variable>
    <!-- fullcalendar view for desktop -->
    <xsl:variable name="defaultDesktopView">
        <xsl:choose>
            <xsl:when test="ametys:serviceViewParameter('defaultDesktopView') != ''"><xsl:value-of select="ametys:serviceViewParameter('defaultDesktopView')"/></xsl:when>
            <xsl:otherwise>dayGridMonth</xsl:otherwise>
        </xsl:choose>
    </xsl:variable>
    <!-- fullcalendar view for mobile -->
    <xsl:variable name="defaultMobileView" select="ametys:serviceViewParameter('defaultMobileView')"/>
    <!-- determines how far forward the scroll pane is initially scrolled -->
    <xsl:variable name="scrollTime">09:00:00</xsl:variable>
    <!-- Determines the first time slot that will be displayed for each day -->
    <xsl:variable name="slotMinTime">00:00:00</xsl:variable>
    <!-- Determines the last time slot that will be displayed for each day -->
    <xsl:variable name="slotMaxTime">24:00:00</xsl:variable>
    <!-- height of the entire calendar. Set 'auto' to assume a natural height and no scrollbars  -->
    <xsl:variable name="calendarHeight">
        <xsl:choose>
            <xsl:when test="ametys:serviceViewParameter('calendarHeight') != '' and ametys:serviceViewParameter('calendarHeight') != '0'"><xsl:value-of select="ametys:serviceViewParameter('calendarHeight')"/></xsl:when>
            <xsl:otherwise>'auto'</xsl:otherwise>
        </xsl:choose>
    </xsl:variable>
    
    <!-- Include Saturday/Sunday columns -->
    <xsl:variable name="displayWeekends">
        <xsl:choose>
            <xsl:when test="ametys:serviceViewParameter('displayWeekends') != ''"><xsl:value-of select="ametys:serviceViewParameter('displayWeekends')"/></xsl:when>
            <xsl:otherwise>true</xsl:otherwise>
        </xsl:choose>
    </xsl:variable>
    
    <!-- dayGridMonth / timeGridWeek / timeGridDay / listMonth -->
    <xsl:variable name="availableViews" select="ametys:serviceViewParameter('calendarViews')"/>
    
    <xsl:variable name="forceAllDay">
        <xsl:choose>
            <xsl:when test="ametys:serviceViewParameter('forceAllDay') != ''">
                <xsl:value-of select="ametys:serviceViewParameter('forceAllDay')"/>
            </xsl:when>
            <xsl:otherwise>false<!-- Default value --></xsl:otherwise>
        </xsl:choose>
    </xsl:variable>
    
    <xsl:variable name="common-service-css-class-name">service-full-calendar</xsl:variable>
    
    <xsl:template name="common-service-head-css">
        <xsl:call-template name="fullcalendar-css"/>
    </xsl:template>
    
    <xsl:template name="fullcalendar-css">
    </xsl:template>
    
    <xsl:template name="common-service-head-js">
        <xsl:call-template name="calendar-search-service-head-js"/>
        
        <xsl:call-template name="fullcalendar-js"/>
    </xsl:template>
    
    <xsl:template name="fullcalendar-js">
        <script type="text/javascript" src="{ametys:pluginResourceURL('fullcalendar', 'js/main.js')}"></script>
        <script type="text/javascript" src="{ametys:pluginResourceURL('fullcalendar', 'js/locales-all.js')}"></script>
        <script type="text/javascript" src="{ametys:pluginResourceURL('calendar', 'js/vendor/popper.js/popper.js')}"></script>
        <script type="text/javascript" src="{ametys:pluginResourceURL('calendar', 'js/vendor/tooltip.js/tooltip.js')}"></script>
        
        <script type="text/javascript">
            
            var calendar_<xsl:value-of select="$serviceId"/>;
            var submit_<xsl:value-of select="$serviceId"/> = false;
            var defaultView_<xsl:value-of select="$serviceId"/> = '<xsl:value-of select="$defaultDesktopView"/>';
            var defaultMobileView_<xsl:value-of select="$serviceId"/> = '<xsl:value-of select="$defaultMobileView"/>' || '<xsl:value-of select="$defaultDesktopView"/>';
            
            var switchViewButtons = [];
            <xsl:for-each select="$availableViews">
                switchViewButtons.push("<xsl:value-of select="."/>");
            </xsl:for-each>
            
            function onSubmit_<xsl:value-of select="$serviceId"/>()
            {
                submit_<xsl:value-of select="$serviceId"/> = true;
                calendar_<xsl:value-of select="$serviceId"/>.refetchEvents();
            }
            
            function showHideAllDayEl_<xsl:value-of select="$serviceId"/>(calendar)
            {
                const viewType = calendar.view.type,
                      $allDayEl = document.querySelector('#calendar-<xsl:value-of select="$serviceId"/> .fc-daygrid-body.fc-daygrid-body-unbalanced.fc-daygrid-body-natural');
                
                if (viewType === 'timeGridWeek' || viewType === 'timeGridDay')
                {
                    const s = calendar.view.activeStart, e = calendar.view.activeEnd;
                    const allDayVisibleEvents = calendar.getEvents().filter(event => {
                        if (!event.end &amp;&amp; s &lt;= event.start &amp;&amp; event.start &lt; e)
                        {
                            return event.allDay;
                        }
                        else if (event.start &lt; e &amp;&amp; s &lt; event.end)
                        {
                            return event.allDay;
                        }
            
                        return false;
                    })
                    
                    // "all day" header will be visible only if there is at least an full day event visible
                    const visibility = allDayVisibleEvents.length > 0 ? 'unset' : 'collapse';
                    $allDayEl ? $allDayEl.style.visibility = visibility : null
                }
            }
           
            $j(document).ready(function () {
            
                // Hide ICS link
                $j("#calendar-search-service-<xsl:value-of select="$serviceId"/> [data-ametys-search-role='ics-link']").hide();
                
                var calendarEl = document.getElementById('calendar-<xsl:value-of select="$serviceId"/>');
                calendar_<xsl:value-of select="$serviceId"/> = new FullCalendar.Calendar(calendarEl, {
                    initialView: window.innerWidth &lt; <xsl:value-of select="$mobileMaxResolution"/> ? defaultMobileView_<xsl:value-of select="$serviceId"/> : defaultView_<xsl:value-of select="$serviceId"/>,
                    timeZone: 'local',
                    dayMaxEvents: <xsl:value-of select="$dayMaxEvents"/>,
                    fixedWeekCount: false,
                    locale: '<xsl:value-of select="$lang"/>',
                    scrollTime: '<xsl:value-of select="$scrollTime"/>',
                    slotMinTime: '<xsl:value-of select="$slotMinTime"/>',
                    slotMaxTime: '<xsl:value-of select="$slotMaxTime"/>',
                    height: <xsl:value-of select="$calendarHeight"/>,
                    weekNumbers: true,
                    weekText: '<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_WEEK_TEXT"/>', // Custom week text
                    weekends: '<xsl:value-of select="$displayWeekends"/>' == 'true', // Whether to include Saturday/Sunday columns in any of the calendar views.
                    navLinks: switchViewButtons.indexOf('timeGridWeek') != -1 &amp;&amp; switchViewButtons.indexOf('timeGridDay') != -1, // enable nav links on day / week only if timeGridWeek and timeGridDay is available
                    /* Accessibility */
                    eventInteractive: true,
                    navLinkHint: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_NAV_LINK_GOTO_HINT" i18n:catalogue="plugin.calendar"/>",
                    buttonHints: {
                        next: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_NAV_LINK_NEXT_HINT" i18n:catalogue="plugin.calendar"/>",
                        prev: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_NAV_LINK_PREV_HINT" i18n:catalogue="plugin.calendar"/>",
                        today: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_TODAY_LINK_HINT" i18n:catalogue="plugin.calendar"/>",
                    },
                    viewHint: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_VIEW_LINK_HINT" i18n:catalogue="plugin.calendar"/>",
                    moreLinkHint: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_MORE_LINK_HINT" i18n:catalogue="plugin.calendar"/>",
                    closeHint: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_CLOSE_HINT" i18n:catalogue="plugin.calendar"/>",
                    /* END accessibility */
                    headerToolbar: {
                        left: 'prev,next today' + (switchViewButtons.length > 1 ? ' ' + switchViewButtons.join(',') : ''),
                        center: 'title',
                        right: '<xsl:if test="ametys:serviceViewParameter('ical-download') = 'true'">icsDownload</xsl:if>'
                    },
                    customButtons: {
                         // Download to ICS format
                         icsDownload: {
                             hint: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_ICS_DOWNLOAD" i18n:catalogue="plugin.calendar"/>",
                             text: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_ICS_DOWNLOAD_SHORT" i18n:catalogue="plugin.calendar"/>",
                            // icon: 'download',
                             click: function() {
                                let icsUrl = $j("#calendar-search-service-<xsl:value-of select="$serviceId"/> [data-ametys-search-role='ics-link']").attr("href");
                                window.open(icsUrl);
                             }
                         }
                    },
                    buttonText: {
                        list: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_BTN_VIEW_LIST"/>"
                    },
                    allDayText: "<i18n:text i18n:key="CALENDAR_SERVICE_SEARCH_CALENDAR_FULLCALENDAR_ALL_DAY_TEXT" i18n:catalogue="plugin.calendar"/>",
                    eventDisplay: 'block',
                    dayHeaderContent: date => {
                        // Custom day header like "mon. 5" for time grid week
                        var activeView = calendar_<xsl:value-of select="$serviceId"/>.view.type;
                        return activeView === 'timeGridWeek' ? $j.datepicker.formatDate('D d', date.date) : date.text
                    },
                    datesSet: dateInfo => {
                        showHideAllDayEl_<xsl:value-of select="$serviceId"/>(dateInfo.view.calendar);
                    },
                    eventDidMount: function (info) {
                        var tooltip = new Tooltip(info.el, {
                            title: info.event.extendedProps.description,
                            html: true,
                            placement: 'top',
                            trigger: 'hover',
                            container: 'body'
                        });
                    },
                    views: {
                        timeGridWeek: { 
                          displayEventTime: false
                        }
                    },
                    events: function( fetchInfo, successCallback, failureCallback ) { 
                    
                        CalendarSearchService.launchSearch({
                                selector: "#calendar-search-service-<xsl:value-of select="$serviceId"/>",
                                siteUriPrefix: "<xsl:value-of select="$site-uri-prefix"/>",
                                zoneItemId: "<xsl:value-of select="/search/zone-item/@id"/>",
                                siteName: "<xsl:value-of select="$site"/>",
                                lang: "<xsl:value-of select="$lang"/>",
                                isSubmit: submit_<xsl:value-of select="$serviceId"/>
                            }, 
                            fetchInfo.startStr, 
                            fetchInfo.endStr,
                            function() {
                                // on success
                                var results = JSON.parse($j('#calendar-search-service-<xsl:value-of select="$serviceId"/> [data-ametys-search-role="json-hits"]').html());
                                successCallback(results.events);
                                
                                submit_<xsl:value-of select="$serviceId"/> = false;
                                
                                showHideAllDayEl_<xsl:value-of select="$serviceId"/>(calendar_<xsl:value-of select="$serviceId"/>);
                            },
                            function() {
                                // on failure
                                submit_<xsl:value-of select="$serviceId"/> = false;
                                failureCallback();
                            }
                        );
                    }

                });
                calendar_<xsl:value-of select="$serviceId"/>.render();
                
                if (defaultView_<xsl:value-of select="$serviceId"/> !== defaultMobileView_<xsl:value-of select="$serviceId"/>)
                {
                    $j(window).resize(function() {
                        var mobileMaxResolution = <xsl:value-of select="$mobileMaxResolution"/>;
                        var currentView = calendar_<xsl:value-of select="$serviceId"/>.view.type;
                        
                        if (window.innerWidth &lt; mobileMaxResolution &amp;&amp; currentView != defaultMobileView_<xsl:value-of select="$serviceId"/>)
                        {
                            calendar_<xsl:value-of select="$serviceId"/>.changeView(defaultMobileView_<xsl:value-of select="$serviceId"/>);
                        }
                        else if (window.innerWidth &gt;= mobileMaxResolution &amp;&amp; currentView != defaultView_<xsl:value-of select="$serviceId"/>)
                        {
                            calendar_<xsl:value-of select="$serviceId"/>.changeView(defaultView_<xsl:value-of select="$serviceId"/>);
                        }
                    });
                }
            });
        </script>
    </xsl:template>
    
    <xsl:template match="hit[content]" mode="json-allday">
        <xsl:choose>
            <xsl:when test="$forceAllDay = 'true'">
                <xsl:text>true</xsl:text>
            </xsl:when>
            <xsl:when test="$forceAllDayIfMidnight and (dateutils:isAtMidnight(content/attributes/start-date, '') and (not(content/attributes/end-date) or dateutils:isAtMidnight(content/attributes/end-date, '')))"><!-- consider as full day event -->
                <xsl:text>true</xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:text>false</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template match="hit[@icsEvent = 'true']" mode="json-allday">
        <xsl:choose>
            <xsl:when test="$forceAllDay = 'true'">
                <xsl:text>true</xsl:text>
            </xsl:when>
            <xsl:when test="event/allDay = 'true'">
                <xsl:text>true</xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:text>false</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="search-js-submit-button">
        <xsl:param name="callback-function" select="$callback-function"/>
        
        <xsl:variable name="code">
            onSubmit_<xsl:value-of select="$serviceId"/>();
            return false;
        </xsl:variable>
        <xsl:value-of select="normalize-space($code)"/>
    </xsl:template>
    
</xsl:stylesheet>