/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.icsreader;

import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.calendar.events.EventsFilterHelper;
import org.ametys.plugins.calendar.icsreader.IcsReader.IcsEvents;
import org.ametys.plugins.calendar.search.CalendarSearchService;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.web.repository.page.ZoneItem;

import net.fortuna.ical4j.model.Parameter;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.parameter.TzId;
import net.fortuna.ical4j.model.property.Created;
import net.fortuna.ical4j.model.property.DtEnd;
import net.fortuna.ical4j.model.property.DtStamp;
import net.fortuna.ical4j.model.property.DtStart;
import net.fortuna.ical4j.model.property.LastModified;
import net.fortuna.ical4j.model.property.Url;

/**
 * Helper from ICS events
 */
public class IcsEventHelper implements Component, Serviceable
{
    /** The component role. */
    public static final String ROLE = IcsEventHelper.class.getName();
    
    /** The ICS reader */
    protected IcsReader _icsReader;
    /** The tag provider extention point */
    protected TagProviderExtensionPoint _tagProviderEP;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _icsReader = (IcsReader) smanager.lookup(IcsReader.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) smanager.lookup(TagProviderExtensionPoint.ROLE);
    }
    
    /**
     * Read the configured distant ics sources into a list of {@link IcsEvents}
     * @param zoneItem zoneItem where the configuration will be fetched
     * @param siteName name of the current site
     * @param dateRange range of dates to limit
     * @return a list of {@link IcsEvents}
     */
    public List<IcsEvents> getICSEvents(ZoneItem zoneItem, String siteName, EventsFilterHelper.DateTimeRange dateRange)
    {
        List<IcsEvents> icsEvents = new ArrayList<>();
        if (zoneItem != null && zoneItem.getServiceParameters().hasValue("ics"))
        {
            Long nbIcsEvent = zoneItem.getServiceParameters().getValue("nbEvents");
            Long maxIcsSize = zoneItem.getServiceParameters().getValue("maxSize");
    
            icsEvents = getICSEvents(zoneItem, siteName, dateRange, nbIcsEvent, maxIcsSize);
        }
        return icsEvents;
    }
    
    /**
     * Read the configured distant ics sources into a list of {@link IcsEvents}
     * @param zoneItem zoneItem where the configuration will be fetched
     * @param siteName name of the current site
     * @param dateRange range of dates to limit
     * @param nbEvents number of events to read
     * @param maxFileSize max ics file size (in bytes)
     * @return a list of {@link IcsEvents}
     */
    public List<IcsEvents> getICSEvents(ZoneItem zoneItem, String siteName, EventsFilterHelper.DateTimeRange dateRange, Long nbEvents, Long maxFileSize)
    {
        List<IcsEvents> icsEvents = new ArrayList<>();

        if (zoneItem.getServiceParameters().hasValue("ics"))
        {
            ModifiableModelAwareRepeater icsRepeater = zoneItem.getServiceParameters().getValue("ics");
            
            for (ModifiableModelAwareRepeaterEntry repeaterEntry : icsRepeater.getEntries())
            {
                String url = repeaterEntry.getValue("url");
                IcsEvents eventList = _icsReader.getEventList(url, dateRange, nbEvents, maxFileSize);
                
                String tagName = repeaterEntry.getValue("tag");
                if (StringUtils.isNotEmpty(tagName))
                {
                    Tag tag = _tagProviderEP.getTag(tagName, Map.of("siteName", siteName));
                    if (tag != null)
                    {
                        eventList.setTag(tag);
                    }
                }

                icsEvents.add(eventList);
            }
        }
        
        return icsEvents;
    }
    
    /**
     * Get the ICS tags from search service
     * @param zoneItem The zone item id
     * @param siteName the site name
     * @return the ICS tags
     */
    public Set<Tag> getIcsTags(ZoneItem zoneItem, String siteName)
    {
        Set<Tag> tags = new LinkedHashSet<>();
        
        if (zoneItem != null && zoneItem.getServiceParameters().hasValue("ics"))
        {
            ModifiableModelAwareRepeater icsRepeater = zoneItem.getServiceParameters().getValue("ics");
            for (ModifiableModelAwareRepeaterEntry repeaterEntry : icsRepeater.getEntries())
            {
                String tagName = repeaterEntry.getValue("tag");
                Tag tag = _tagProviderEP.getTag(tagName, Map.of("siteName", siteName));
                if (tag != null)
                {
                    tags.add(tag);
                }
            }
        }
        
        return tags;
    }
    
    /**
     * Get a list of {@link LocalVEvent} form the list of {@link IcsEvents}
     * @param icsEventsList the list of {@link IcsEvents}
     * @param dateRange range of dates to limit
     * @return a list of {@link LocalVEvent}
     */
    public Pair<List<LocalVEvent>, String> toLocalIcsEvent(List<IcsEvents> icsEventsList, EventsFilterHelper.DateTimeRange dateRange)
    {
        return toLocalIcsEvent(icsEventsList, dateRange, List.of());
    }
    
    /**
     * Get a list of {@link LocalVEvent} form the list of {@link IcsEvents}
     * @param icsEventsList the list of {@link IcsEvents}
     * @param dateRange range of dates to limit
     * @param filteredTags A list of tag's name to filter ICS events. Can be empty to no filter on tags.
     * @return a list of {@link LocalVEvent}
     */
    public Pair<List<LocalVEvent>, String> toLocalIcsEvent(List<IcsEvents> icsEventsList, EventsFilterHelper.DateTimeRange dateRange, List<String> filteredTags)
    {
        List<LocalVEvent> localICSEvents = new ArrayList<>();
        String fullICSDistantEvents = "";
        for (IcsEvents icsEvents : icsEventsList)
        {
            if (icsEvents.hasEvents())
            {
                Tag tag = icsEvents.getTag();
                if (filteredTags.isEmpty() || tag != null && filteredTags.contains(tag.getName()))
                {
                    for (VEvent calendarComponent : icsEvents.getEvents())
                    {
                        List<LocalVEvent> localCalendarComponents = _icsReader.getEventDates(calendarComponent, dateRange, tag);
                        localICSEvents.addAll(localCalendarComponents);
                        if (dateRange == null)
                        {
                            // To avoid to have the ICS events in double in some exotic non-anticipated cases, when there is a dateRange, we do not copy the ICS
                            fullICSDistantEvents += calendarComponent.toString() + "\n";
                        }
                    }
                }
            }
        }
        
        return new ImmutablePair<>(localICSEvents, fullICSDistantEvents);
    }
    /**
     * Get a list of {@link LocalVEvent} form the list of {@link IcsEvents}
     * @param icsEventsList the list of {@link IcsEvents}
     * @param dateRange range of dates to limit
     * @param filteredTags A list of tag's name to filter ICS events. Can be empty to no filter on tags.
     * @return a list of {@link LocalVEvent}
     */
    public String toVTimeZone(List<IcsEvents> icsEventsList, EventsFilterHelper.DateTimeRange dateRange, List<String> filteredTags)
    {
        Set<String> timeZoneIds = new HashSet<>();
        String timeZonesAsString = "";
        for (IcsEvents icsEvents : icsEventsList)
        {
            if (icsEvents.hasEvents())
            {
                Tag tag = icsEvents.getTag();
                if (filteredTags.isEmpty() || tag != null && filteredTags.contains(tag.getName()))
                {
                    for (VEvent calendarComponent : icsEvents.getEvents())
                    {
                        DtStart<Temporal> dateTimeStart = calendarComponent.getDateTimeStart();
                        TzId startDateTZ = dateTimeStart != null ? dateTimeStart.<TzId>getParameter(Parameter.TZID).orElse(null) : null;
                        if (startDateTZ != null)
                        {
                            if (!timeZoneIds.contains(startDateTZ.getValue()))
                            {
                                timeZoneIds.add(startDateTZ.getValue());

                                timeZonesAsString += startDateTZ.getValue() + "\n";
                            }
                        }
                        DtEnd<Temporal> dateTimeEnd = calendarComponent.getDateTimeEnd();
                        TzId endDateTZ = dateTimeEnd != null ? dateTimeEnd.<TzId>getParameter(Parameter.TZID).orElse(null) : null;
                        if (endDateTZ != null)
                        {
                            if (!timeZoneIds.contains(endDateTZ.getValue()))
                            {
                                timeZoneIds.add(endDateTZ.getValue());

                                timeZonesAsString += endDateTZ.getValue() + "\n";
                            }
                        }
                    }
                }
            }
        }
        
        return timeZonesAsString;
    }
    
    /**
     * SAX ics events hits
     * @param handler the content handler
     * @param icsEvents The ics events
     * @param startNumber the start index
     * @throws SAXException if an error occurred while saxing
     */
    public void saxIcsEventHits(ContentHandler handler, List<LocalVEvent> icsEvents, int startNumber) throws SAXException
    {
        int hitIndex = startNumber;
        for (LocalVEvent icsEvent : icsEvents)
        {
            saxIcsEventHit(handler, icsEvent, hitIndex++);
        }
    }
    
    /**
     * SAX a ics events hit
     * @param handler the content handler
     * @param icsEvent The ics event
     * @param number the hit index
     * @throws SAXException if an error occurred while saxing
     */
    public void saxIcsEventHit(ContentHandler handler, LocalVEvent icsEvent, int number) throws SAXException
    {
        VEvent event = icsEvent.getEvent();
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("number", Integer.toString(number));
        attrs.addCDATAAttribute("icsEvent", "true");
        XMLUtils.startElement(handler, "hit", attrs);

        String id = event.getProperty(Property.UID).map(Property::getValue).orElse(UUID.randomUUID().toString());
        XMLUtils.createElement(handler, "id", id);
        
        saxIcsEvent(handler, icsEvent);
        
        XMLUtils.endElement(handler, "hit");
    }
    
    /**
     * SAX a ics event
     * @param handler the content handler
     * @param icsEvent The ics event
     * @throws SAXException if an error occurred while saxing
     */
    public void saxIcsEvent(ContentHandler handler, LocalVEvent icsEvent) throws SAXException
    {
        XMLUtils.startElement(handler, "event");
        
        VEvent event = icsEvent.getEvent();
        
        String title = event.getProperty(Property.SUMMARY).map(Property::getValue).orElse(StringUtils.EMPTY);
        XMLUtils.createElement(handler, "title", title);
        
        String description = event.getProperty(Property.DESCRIPTION).map(Property::getValue).orElse(StringUtils.EMPTY);
        XMLUtils.createElement(handler, "description", description);
        
        DtStamp dateTimeStamp = event.getDateTimeStamp();
        Instant dtStamp = dateTimeStamp != null ? dateTimeStamp.getDate() : Instant.now();
        
        Created created = event.getCreated();
        Instant creationDate = created != null ? created.getDate() : dtStamp;
        
        LastModified lastModified = event.getLastModified();
        Instant lastModifiedDate = lastModified != null ? lastModified.getDate() : dtStamp;

        _saxInstant(handler, "creationDate", creationDate);
        _saxInstant(handler, "lastModifiedDate", lastModifiedDate);
        
        _saxZonedDateTime(handler, "startDate", icsEvent.getStart());
        _saxZonedDateTime(handler, "endDate", icsEvent.getEnd());
        
        _saxAllDay(handler, icsEvent);
        
        CalendarSearchService.saxTag(handler, icsEvent.getTag());
        
        Url url = event.getUrl();
        if (url != null)
        {
            XMLUtils.createElement(handler, "url", url.getValue());
        }
        
        XMLUtils.endElement(handler, "event");
        
    }
    
    /**
     * Sax all day property
     * @param handler the content handler
     * @param icsEvent the ICS event
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxAllDay(ContentHandler handler, LocalVEvent icsEvent) throws SAXException
    {
        VEvent event = icsEvent.getEvent();
        
        DtStart startDate = event.getDateTimeStart();
        DtEnd endDate = event.getDateTimeEnd();
        
        boolean allDayEvent = startDate != null && "DATE".equals(startDate.getParameter("VALUE").map(Parameter::getValue).orElse(null))
                    && (endDate == null || "DATE".equals(endDate.getParameter("VALUE").map(Parameter::getValue).orElse(null)));
        
        XMLUtils.createElement(handler, "allDay", String.valueOf(allDayEvent));
    }
    
    /**
     * Sax ICS date
     * @param handler the content handler
     * @param tagName the xml tag name
     * @param zonedDatetime the ZonedDateTime to sax
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxZonedDateTime(ContentHandler handler, String tagName, ZonedDateTime zonedDatetime) throws SAXException
    {
        if (zonedDatetime != null)
        {
            XMLUtils.createElement(handler, tagName, DateUtils.getISODateTimeFormatter().format(zonedDatetime));
        }
    }

    /**
     * Sax ICS date
     * @param handler the content handler
     * @param tagName the xml tag name
     * @param instant the Instant to sax
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxInstant(ContentHandler handler, String tagName, Instant instant) throws SAXException
    {
        if (instant != null)
        {
            _saxZonedDateTime(handler, tagName, instant.atZone(ZoneOffset.UTC));
        }
    }

    /**
     * Sax ICS errors
     * @param icsResults the ICS events
     * @param handler the content handler
     * @throws SAXException if an error occurred
     */
    public void saxICSErrors(List<IcsEvents> icsResults, ContentHandler handler) throws SAXException
    {
        List<IcsEvents> errorsIcs = icsResults.stream()
            .filter(ics -> ics.getStatus() != IcsEvents.Status.OK)
            .collect(Collectors.toList());

        if (!errorsIcs.isEmpty())
        {
            XMLUtils.startElement(handler, "errors-ics");
            for (IcsEvents ics : errorsIcs)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("url", ics.getUrl());
                attrs.addCDATAAttribute("status", ics.getStatus().name());
                XMLUtils.createElement(handler, "ics", attrs);
            }
            XMLUtils.endElement(handler, "errors-ics");
        }
    }
}
